﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Windows.Forms;

namespace FontEdit
{
	//フォントサイズ構造体
	public struct FontSize {
		public int	nPoint;
		public int	nXres;
		public int	nYres;
	};
	//バウンディングボックス構造体
	public struct BBox {
		public int	nBBw;
		public int	nBBh;
		public int	nXoff;
		public int	nYoff;
	};
	//フォント情報構造体
	public struct FontInfo {
		public string	sFoundry;
		public string	sFamily;
		public string	sWeight;
		public string	sSlant;
		public string	sSetwidth;
		public string	sSpacing;
		public string	sCharset;
	};

	//BDFフォントのヘッダ部クラス
	public class BdfHead : IDisposable {
		public String		sVersion;
		public List<String>	slCmnt;
		public String		sFont;
		public FontSize		stSize;
		public BBox			stFontBBX;
		public List<String>	slProperty;
		public int			nChars;

		//コンストラクタ
		public BdfHead()
		{
			slCmnt = new List<String>();
			slProperty = new List<String>();
			Clear();
		}
		//コピーコンストラクタ
		public BdfHead(BdfHead hdr)
		{
			sVersion = hdr.sVersion;
			slCmnt = new List<String>(hdr.slCmnt);
			sFont = hdr.sFont;
			stSize = hdr.stSize;
			stFontBBX = hdr.stFontBBX;
			slProperty = new List<String>(hdr.slProperty);
			nChars = hdr.nChars;
		}
		//データ破棄
		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}
		//データ破棄
		protected virtual void Dispose(bool disposing)
		{
			if (disposing) {
				if (slCmnt != null) {
					slCmnt.Clear();
				}
				if (slProperty != null) {
					slProperty.Clear();
				}
				slCmnt = null;
				slProperty = null;
			}
		}
		//デストラクタ
		~BdfHead()
		{
			Dispose(false);
		}
		//データクリア
		public void Clear()
		{
			sVersion = "";
			sFont = "";
			stSize.nPoint = 0;
			stSize.nXres = 0;
			stSize.nYres = 0;
			stFontBBX.nBBw = 0;
			stFontBBX.nBBh = 0;
			stFontBBX.nXoff = 0;
			stFontBBX.nYoff = 0;
			nChars = 0;

			slProperty.Clear();
			slCmnt.Clear();
		}
	};

	//BDFフォントの文字情報ヘッダ部クラス
	public class CharHead {
		public String	sStartChar;
		public int		nEncode;
		public int		nSWidth_x;
		public int		nSWidth_y;
		public int		nDWidth_x;
		public int		nDWidth_y;
		public BBox		stBBX;

		//コンストラクタ
		public CharHead()
		{
			Clear();
		}
		//コピーコンストラクタ
		public CharHead(CharHead hdr)
		{
			sStartChar = hdr.sStartChar;
			nEncode = hdr.nEncode;
			nSWidth_x = hdr.nSWidth_x;
			nSWidth_y = hdr.nSWidth_y;
			nDWidth_x = hdr.nDWidth_x;
			nDWidth_y = hdr.nDWidth_y;
			stBBX = hdr.stBBX;
		}
		//データクリア
		public void Clear()
		{
			sStartChar = "";
			nEncode = 0;
			nSWidth_x = 0;
			nSWidth_y = 0;
			nDWidth_x = 0;
			nDWidth_y = 0;
			stBBX.nBBw = 0;
			stBBX.nBBh = 0;
			stBBX.nXoff = 0;
			stBBX.nYoff = 0;
		}
	};

	//フォントデータクラス
	public class FontData
	{
		private	List<byte[]>	m_lstFontList;
		private String			m_sFileName;
		private int				m_nWidth;
		private int				m_nHeight;
		private int				m_nLineBytes;
		private int				m_nDataSize;
		private bool			m_bEnable;
		BdfHead					m_BdfHdr;
		List<CharHead>			m_CharHdr;
		FontInfo				m_FontInf;

		public const int		POINT_PER_INCH = 72;
		public const int		PIXEL_PER_INCH = 75; 

		//コンストラクタ
		public FontData()
		{
			m_sFileName = "";
			m_nWidth = 0;
			m_nHeight = 0;
			m_nLineBytes = 0;
			m_nDataSize = 0;
			m_bEnable = false;

			m_lstFontList = new List<byte[]>();
			m_BdfHdr = new BdfHead();
			m_CharHdr = new List<CharHead>();

			m_FontInf.sFoundry = "Misc";
			m_FontInf.sFamily = "Fixed";
			m_FontInf.sWeight = "Medium";
			m_FontInf.sSlant = "R";
			m_FontInf.sSetwidth = "Normal";
			m_FontInf.sSpacing = "M";
			m_FontInf.sCharset = "ISO8859";
		}

		//デストラクタ
		~FontData()
		{
			if (m_lstFontList != null) {
				m_lstFontList.Clear();
			}
			if (m_BdfHdr != null) {
				m_BdfHdr.Clear();
			}
			if (m_CharHdr != null) {
				m_CharHdr.Clear();
			}
			m_lstFontList = null;
			m_BdfHdr = null;
			m_CharHdr = null;
		}

		//文字データ数取得
		public int	Count()
		{
			return m_lstFontList.Count;
		}
		//ビットマップデータのサイズ（バイト数）取得
		public int	GetDataSize()
		{
			return m_nDataSize;
		}
		//横ピクセル数取得
		public int	GetWidth()
		{
			return m_nWidth;
		}
		//縦ピクセル数取得
		public int	GetHeight()
		{
			return m_nHeight;
		}
		//1ラインのバイト数取得
		public int	GetLineBytes()
		{
			return m_nLineBytes;
		}
		//フォント全体のバウンディングボックス取得
		public BBox	GetFontBBox()
		{
			return m_BdfHdr.stFontBBX;
		}
		//フォント情報を取得
		public FontInfo GetFintInfo()
		{
			return m_FontInf;
		}

		//フォントデータを生成する
		public bool	Create(int width, int height, int line_bytes)
		{
			Delete();

			int		bytes;
			double	point;
			string	str;
			int		ascent, descent;

			bytes = width / 8;
			if (width % 8 > 0) {
				bytes++;
			}
			if (line_bytes < bytes) {
				line_bytes = bytes;
			}
			m_nWidth = width;
			m_nHeight = height;
			m_nLineBytes = line_bytes;
			m_nDataSize = m_nLineBytes * m_nHeight;

			//ポイント数を計算
			point = (double)POINT_PER_INCH * m_nHeight / PIXEL_PER_INCH;

			ascent = m_nHeight;
			descent = 0;

			//フォーマットバージョン
			m_BdfHdr.sVersion = "2.1";

			//ポイント数
			m_BdfHdr.stSize.nPoint = (int)point;
			if ((double)m_BdfHdr.stSize.nPoint < point) {
				m_BdfHdr.stSize.nPoint++;
			}
			//解像度情報
			m_BdfHdr.stSize.nXres = PIXEL_PER_INCH;
			m_BdfHdr.stSize.nYres = PIXEL_PER_INCH;

			//フォント全体のバウンディングボックス
			m_BdfHdr.stFontBBX.nBBw = m_nWidth;
			m_BdfHdr.stFontBBX.nBBh = m_nHeight;
			m_BdfHdr.stFontBBX.nXoff = 0;
			m_BdfHdr.stFontBBX.nYoff = 0;

			//論理フォント名
			m_BdfHdr.sFont = "";

			//プロパティのフォントDISCENT
			str = "FONT_DESCENT " + descent.ToString();
			m_BdfHdr.slProperty.Add(str);

			//プロパティのフォントASCENT
			str = "FONT_ASCENT " + ascent.ToString();
			m_BdfHdr.slProperty.Add(str);

			m_BdfHdr.nChars = 0;

			m_bEnable = true;

			return true;
		}

		//フォントデータを破棄する
		public void	Delete()
		{
			int	i;

			for (i = m_lstFontList.Count - 1; i >= 0; i--) {
				m_lstFontList[i] = null;
				m_lstFontList.RemoveAt(i);
			}
			for (i = m_CharHdr.Count - 1; i >= 0; i--) {
				m_CharHdr[i] = null;
				m_CharHdr.RemoveAt(i);
			}
			m_CharHdr.Clear();

			m_BdfHdr.Clear();

			m_nWidth = 0;
			m_nHeight = 0;
			m_nLineBytes = 0;
			m_nDataSize = 0;

			m_bEnable = false;
		}

		//C/C++配列形式のフォントファイルを読み込む
		public bool Load(String fname, int width, int height, int line_bytes)
		{
            string			line = "";
			int				n, i, num;
			int				fnt_num;
			string[]		items;
			string			sVal;
			string			sName;
			StreamReader	file;
			int				idx;
			bool			bData;
			bool			bComment;
			bool			bStart;
			string			key;
			string			val;
			int				w, h, lb;
			char[]			blank = { ' ', '\t' };
			char[]			delimit = { ' ', '\t', ',', ':' };

			fnt_num = 0;
			n = 0;
			w = -1;
			h = -1;
			lb = -1;

			try {
	            file = new StreamReader(fname);

				Delete();
				Create(width, height, line_bytes);

				bStart = false;
				bComment = false;
				bData = false;
				sName = "";

				while ((line = file.ReadLine()) != null) {
					line = line.TrimStart(blank);
					if (line.Length < 2) {
						continue;
					}
					if (line.Substring(0, 2) == "/*") {
						bComment = true;
						line = line.Substring(2).TrimStart(blank);
						if (line.Length == 0) {
							continue;
						}
					}
					if (!bStart) {	//データ部が始まる前
						//ブランクで区切られた2つのワードがあるか
						idx = line.IndexOfAny(blank);
						if (idx > 0) {
							//先頭のワード
							key = line.Substring(0, idx);

							//2番目のワード
							val = line.Substring(idx);
							val = val.TrimStart(blank);
							if (val.Length <= 0) {
								//2番目のワードが無ければ意味をなさないので次の行へ
								continue;
							}
							//2番目のワード以降にワードがある場合は切り捨てる
							idx = val.IndexOfAny(blank);
							if (idx > 0) {
								val = val.Substring(0, idx);
							}
							if (key == "Width") {
								//横ピクセル数
								w = int.Parse(val);
							} else if (key == "Height") {
								//縦ピクセル数
								h = int.Parse(val);
							} else if (key == "LineBytes") {
								//1ラインのバイト数
								lb = int.Parse(val);
							}
						}
					}
					if (bComment) {
						//コメント内
						//コメントの終わりを探す
						idx = line.IndexOf("*/");
						if (idx >= 0) {
							//コメントは終わり
							bComment = false;

							//コメント以降の文字列があれば行データとして取り出す
							line = line.Substring(idx+2).TrimStart(blank);
							if (line.Length == 0) {
								//行データが無ければ次の読込
								continue;
							}
						} else {
							//コメント内なので次の読込へ
							continue;
						}
					}
					if (!bData) {
						//データ部ではない
						idx = line.IndexOf("[");
						if (idx > 0) {
							if (!bStart) {
								bStart = true;
								if ((w > 0 && w != width) || (h > 0 && h != height) ||
														(lb > 0 && lb != line_bytes)) {
									width = w;
									height = h;
									line_bytes = width / 8;
									if (width % 8 > 0) {
										line_bytes++;
									}
									if (lb > line_bytes) {
										line_bytes = lb;
									}
									Delete();
									Create(width, height, line_bytes);
								}
							}
							//'['の直前までの文字列の最後のワードが名前になる
							key = line.Substring(0, idx).TrimEnd(blank);
							items = key.Split(blank);
							sName = items[items.Length - 1];

							//'['の前の文字列を取り除く
							line = line.Substring(idx);
							line = line.TrimStart(blank);

							//']'があったらそこまでの文字列を取り除く
							if ((idx = line.IndexOf("]")) > 0) {
								line = line.Substring(idx + 1);
							}
						}
						//配列のデータ部開始の文字'{'を探す
						idx = line.IndexOf("{");
						if (idx < 0) {
							continue;
						} else {
							bData = true;	//データ部開始

							//文字データ格納用のバッファを追加
							AddFontData(null, sName);
							fnt_num++;
							n= 0;
						}
					} else {
						//データ部
						//配列のデータ部終了の文字'}'を探す
						idx = line.IndexOf("}");
						if (idx >= 0) {
							bData = false;
							//'}'の前までをデータ部文字列として取り出す
							line = line.Substring(0, idx);
						}
					}
					if (bData && line.Length > 0) {
						//データ部で行データがある場合

						//行データをカンマで分割
						items = line.Split(',');
						num = items.Length;

						for (i = 0; i < num && n < m_nDataSize; i++) {
							//各バイトの先頭は0xが付いている
							idx = items[i].IndexOf("0x");
							if (idx >= 0) {
								//先頭の0xを取り除く
								sVal = items[i].Substring(idx + 2, 2);
								if (sVal.Length >= 2) {
									//16進数文字列をバイトデータに変換して格納する
									((byte[])m_lstFontList[fnt_num-1])[n++] = (byte)Convert.ToInt16(sVal, 16);
								}
							}
						}
					}
				}
				file.Close();
			} catch {
				return false;
			}
			if (m_lstFontList.Count > 0) {
				//読み込んだ文字数をヘッダ部位セット
				m_BdfHdr.nChars = m_lstFontList.Count;

				//ファイル名を保存
				m_sFileName = fname;
			}
			return true;
		}

		//C/C++配列形式でフォントファイルを保存する
		public bool Save(String fname, int start = 0, int end = -1)
		{
			if (!m_bEnable) {
				return false;
			}
			int				fnt_num;
			int				i, j, idx;
            string			line = "";
			StreamWriter	file;
			byte[]			fnt = new byte[] { 0x00 };
			string			item;

			fnt_num = m_lstFontList.Count;
			
			if (end < 0) {
				end = fnt_num - 1;
			}
			if (end < start) {
				return false;
			}

			try {
	            file = new StreamWriter(fname, false);

				//フォントサイズをコメント形式で書き込む
				line = "/*";
				file.WriteLine(line);

				line = "Width " + m_nWidth.ToString();
				file.WriteLine(line);

				line = "Height " + m_nWidth.ToString();
				file.WriteLine(line);

				line = "LineBytes " + m_nLineBytes.ToString();
				file.WriteLine(line);

				line = "*/";
				file.WriteLine(line);

				file.WriteLine("");

				for (idx = start; idx <= end; idx++) {
					line = "unsigned char " + m_CharHdr[idx].sStartChar + "[] = {";
					file.WriteLine(line);

					line = "\t";
					j = 0;
					for (i = 0; i < m_nDataSize; i++) {
						fnt[0] = m_lstFontList[idx][i];
						item = "0x" + BitConverter.ToString(fnt);
						if (i < m_nDataSize - 1) {
							item += ",";
						}
						line += item;
						j++;
						if (j >= m_nLineBytes) {
							file.WriteLine(line);
							j = 0;
							line = "\t";
						}
					}
					if (j > 0) {
						file.WriteLine(line);
					}
					file.WriteLine("};");
					file.WriteLine("");
				}
				file.Close();
			} catch {
				return false;
			}
			return true;
		}

		//BDFフォントの全体情報（ヘッダ）部分を読み込む
		public bool	LoadBdfHead(StreamReader file, ref BdfHead hdr)
		{
			bool		res = true;
			char[]		delimiters = { ' ', '\t' };
			string[]	items;
			bool		bProperty = false;
			int			idx;
			string		line = "";
			string		key = "";
			string		val = "";

			try {
				//ファイルから1行ずつ読み込む
				while ((line = file.ReadLine()) != null) {
					//先頭の空白を取り除く
					line = line.TrimStart(delimiters);
					if (line.Length <= 0) {
						//空行は読み飛ばす
						continue;
					}
					//空白文字で分割されたキーワードを探す
					idx = line.IndexOfAny(delimiters);
					if (idx > 0) {
						//途中に空白があるので空白の前をキーワードとして取り出す
						key = line.Substring(0, idx);
						val = line.Substring(idx);
						val = val.TrimStart();
					} else {
						//途中には空白が無いので、行全体がキーワード
						key = line;
						val = "";
					}
					if (bProperty) {	//プロパティ部か
						if (key == "ENDPROPERTIES") {
							//プロパティ部の終了
							bProperty = false;
						} else {
							//プロパティ部の情報は行全体をプロパティ文字列の配列に保存
							hdr.slProperty.Add(line);
						}
					} else {
						if (key == "STARTFONT") {
							//フォント開始（ファイル形式のバージョンを保存）
							hdr.sVersion = val;
						} else if (key == "COMMENT") {
							//コメントはコメント文字列の配列に保存
							hdr.slCmnt.Add(val);
						} else if (key == "FONT") {
							//フォントの論理フォント名を保存
							hdr.sFont = val;
						} else if (key == "SIZE") {
							//フォントのサイズ情報
							items = val.Split(delimiters);
							if (items.Length >= 3) {
								//ポイント数
								hdr.stSize.nPoint = int.Parse(items[0]);
								//X方向解像度(通常は75dpiが使用される)
								hdr.stSize.nXres = int.Parse(items[1]);
								//Y方向解像度(通常は75dpiが使用される)
								hdr.stSize.nYres = int.Parse(items[2]);
							}
						} else if (key == "FONTBOUNDINGBOX") {
							//フォント全体のバウンディングボックス情報
							items = val.Split(delimiters);
							if (items.Length >= 4) {
								//横ピクセル数
								hdr.stFontBBX.nBBw = int.Parse(items[0]);
								//縦ピクセル数
								hdr.stFontBBX.nBBh = int.Parse(items[1]);
								//原点のX方向オフセット
								hdr.stFontBBX.nXoff = int.Parse(items[2]);
								//原点のY方向オフセット
								hdr.stFontBBX.nYoff = int.Parse(items[3]);
							}
						} else if (key == "STARTPROPERTIES") {
							//プロパティ部の開始
							bProperty = true;
						} else if (key == "CHARS") {
							//文字情報の開始
							if (val.Length > 0) {
								//文字数を取得
								hdr.nChars = int.Parse(val);
								break;
							} else {
								//文字数が入っていない
								break;
							}
						} else {
							continue;
						}
					}
				}
			} catch {
				res = false;
			}
			return res;
		}

		//BDFフォントの文字情報部分を読み込む
		public bool	LoadBdfChars(StreamReader file, BdfHead hdr)
		{
			bool		res = true;
            string		line = "";
			char[]		delimiters = { ' ', '\t' };
			string[]	items;
			string		str;
			bool		bBmp = false;
			bool		bChar = false;
			int			idx = -1;
			int			n;
			int			w = m_nWidth;
			int			h = m_nHeight;
			int			xoff = 0;
			int			yoff = 0;
			int			line_bytes = m_nLineBytes;
			string		key;
			string		val;
			int			pos;
			CharHead	chr_hdr = new CharHead();

			n = 0;

			try {
				//ファイルから1行ずつ読み込む
				while (res && (line = file.ReadLine()) != null) {
					//先頭の空白を取り除く
					line = line.TrimStart(delimiters);
					if (line.Length <= 0) {
						//空行は読み飛ばす
						continue;
					}
					//空白文字で分割されたキーワードを探す
					pos = line.IndexOfAny(delimiters);
					if (pos > 0) {
						//途中に空白があるので空白の前をキーワードとして取り出す
						key = line.Substring(0, pos);
						val = line.Substring(pos);
						val = val.TrimStart();
					} else {
						//途中には空白が無いので、行全体がキーワード
						key = line;
						val = "";
					}
					if (!bChar) {
						//文字データ部ではない
						if (key == "STARTCHAR") {
							//文字データ部の開始
							idx++;
							if (val.Length > 0) {
								//文字の名前を取得
								chr_hdr.sStartChar = val;
							} else {
								//文字の名前はインデックス番号
								chr_hdr.sStartChar = idx.ToString();
							}
							bChar = true;
							continue;
						} else {
							//文字データ部は開始していないので次の行を読み込む
							continue;
						}
					} else {
						//文字データ部
						if (!bBmp) {
							//ビットマップデータ部ではない
							if (key == "ENCODING") {
								//エンコーディング番号を取得
								if (val.Length > 0) {
									chr_hdr.nEncode = int.Parse(val);
								} else {
									//記載が無いのでインデックスで代用
									chr_hdr.nEncode = idx;
								}
								continue;
							} else if (key == "SWIDTH") {
								//後続文字の位置情報（スケーリングされた値）
								items = val.Split(delimiters);
								if (items.Length >= 2) {
									//XとYの2ワードが必要
									chr_hdr.nSWidth_x = int.Parse(items[0]);
									chr_hdr.nSWidth_y = int.Parse(items[1]);
								}
								continue;
							} else if (key == "DWIDTH") {
								//後続文字の位置情報（ピクセル数）
								items = val.Split(delimiters);
								if (items.Length >= 2) {
									//XとYの2ワードが必要
									chr_hdr.nDWidth_x = int.Parse(items[0]);
									chr_hdr.nDWidth_y = int.Parse(items[1]);
								}
								continue;
							} else if (key == "BBX") {
								//文字のバウンディングボックス
								items = val.Split(delimiters);
								if (items.Length >= 4) {
									//BBw, BBh, Xoff, Yoff の4ワードが必要
									chr_hdr.stBBX.nBBw = int.Parse(items[0]);
									chr_hdr.stBBX.nBBh = int.Parse(items[1]);
									chr_hdr.stBBX.nXoff = int.Parse(items[2]);
									chr_hdr.stBBX.nYoff = int.Parse(items[3]);

									w = chr_hdr.stBBX.nBBw;		//横ピクセル数
									h = chr_hdr.stBBX.nBBh;		//縦ピクセル数

									//1ラインのバイト数を計算
									line_bytes = w / 8;
									if (w % 8 > 0) {
										line_bytes++;
									}
									//フォント全体のバウンディングボックスと文字のバウンディングボックスを
									//比較して差分をオフセットとして計算する
									xoff = chr_hdr.stBBX.nXoff - hdr.stFontBBX.nXoff;
									yoff = m_nHeight - (h + (chr_hdr.stBBX.nYoff - hdr.stFontBBX.nYoff));
								}
								continue;
							} else if (key == "BITMAP") {
								//ビットマップ情報の開始
								//フォントのデータ格納領域を追加
								AddFontData(null, chr_hdr);
								n = 0;			//ビットマップデータのバイト位置を初期化
								bBmp = true;	//ビットマップデータ部開始
								continue;
							}
						} else {
							//ビットマップデータ部
							if (key == "ENDCHAR") {
								//ビットマップデータ部及び文字データ部の終了
								int	i;
								if (xoff > 0) {
									//全体と比較したX方向のオフセットが＋の場合右方向にシフト
									for (i = 0; i < xoff; i++) {
										ShiftRight(idx);
									}
								} else if (xoff < 0) {
									//全体と比較したX方向のオフセットが－の場合左方向にシフト
									for (i = 0; i > xoff; i--) {
										if (!ShiftLeft(idx)) {
											res = false;
											break;
										}
									}
								}
								if (yoff > 0) {
									//Y全体と比較した方向のオフセットが＋の場合下方向にシフト
									for (i = 0; i < yoff; i++) {
										if (!ShiftDown(idx)) {
											res = false;
											break;
										}
									}
								} else if (yoff < 0) {
									//Y全体と比較した方向のオフセットが－の場合上方向にシフト
									for (i = 0; i > yoff; i--) {
										if (!ShiftUp(idx)) {
											res = false;
											break;
										}
									}
								}
								bBmp = false;	//ビットマップデータ部の終了
								bChar = false;	//文字データ部の終了
							} else {
								//ビットマップデータ
								int	bytes = 0;

								//1ラインのデータを処理する
								while (bytes < line_bytes && key.Length > 0) {
									//2文字づつ取り出す（1バイトのデータ）
									str = key.Substring(0, 2);
									if (str.Length >= 2) {
										//16進数を値に変換してバッファに格納する
										((byte[])m_lstFontList[idx])[n++] = (byte)Convert.ToInt16(str, 16);

										//先頭の2文字を取り除く(次の文字へ)
										key = key.Substring(2);
										bytes++;
									} else {
										break;
									}
								}
								while (bytes < m_nLineBytes) {
									//全体の1行バイト数より少ない場合は
									//ビットマップデータのバイト位置を次のライン迄スキップする
									n++;
									bytes++;
								}
							}
						}
					}
				}
			} catch {
				res = false;
			}

			return res;
		}

		//BDFフォントファイルを読み込む
		public bool	LoadBdf(String fname)
		{
			StreamReader	file;
			BdfHead			hdr = new BdfHead();

			try {
				//フォントファイルを読込モードでオープン
	            file = new StreamReader(fname);

				//BDFフォントの全体情報（ヘッダ）部分を読み込む
				if (!LoadBdfHead(file, ref hdr)) {
					file.Close();
					return false;
				}
				//読み込んだヘッダ情報を元にフォントデータバッファを作成
				if (!Create(hdr.stFontBBX.nBBw, hdr.stFontBBX.nBBh, 0)) {
					file.Close();
					return false;
				}
				//ヘッダ情報を保存
				m_BdfHdr = null;
				m_BdfHdr = new BdfHead(hdr);

				//BDFフォントの文字情報部分を読み込む
				if (!LoadBdfChars(file, m_BdfHdr)) {
					file.Close();
					return false;
				}
				//フォントファイルをクローズ
				file.Close();
			} catch {
				return false;
			}
			hdr = null;

			if (m_lstFontList.Count > 0) {
				string[]	items;

				//実際に読み込んだ文字数をヘッダにセット
				m_BdfHdr.nChars = m_lstFontList.Count;

				items = m_BdfHdr.sFont.Split('-');
				if (items.Length >= 14) {
					m_FontInf.sFoundry = items[1];
					m_FontInf.sFamily = items[2];
					m_FontInf.sWeight = items[3];
					m_FontInf.sSlant = items[4];
					m_FontInf.sSetwidth = items[5];
					m_FontInf.sSpacing = items[11];
					m_FontInf.sCharset = items[13];
				}
				//フォント名を保存
				m_sFileName = fname;
			}
			return true;
		}

		//BDFフォントファイルのヘッダ部を保存する
		public bool	SaveBdfHead(StreamWriter file, BdfHead hdr, FontInfo Inf, int count)
		{
			bool		res = true;
            string		line = "";
			int			i;

			try {
				//フォントのヘッダ情報を出力

				//フォント開始＋フォーマットのバージョンを出力
				line = "STARTFONT " + hdr.sVersion;
				file.WriteLine(line);
				
				//コメント部を出力
				for (i = 0; i < m_BdfHdr.slCmnt.Count; i++) {
					line = "COMMENT " + m_BdfHdr.slCmnt[i];
					file.WriteLine(line);
				}

				//フォントの論理名を出力
				if (hdr.sFont.Length > 0) {
					string[]	items;

					items = hdr.sFont.Split('-');
					if (items.Length >= 14) {
						line = "FONT " + "-" + Inf.sFoundry + "-" + Inf.sFamily + "-" +
								Inf.sWeight + "-" + Inf.sSlant + "-" + Inf.sSetwidth + "-" + 
								items[6] + "-" + items[7] + "-" + items[8] + "-" +
								items[9] + "-" + items[10] + "-" +
								Inf.sSpacing + "-" + items[12] + "-" +
								Inf.sCharset + "-" + items[14];
					} else {
						line = "FONT " + hdr.sFont;
					}
				} else {
					line = "FONT " + "-" + Inf.sFoundry + "-" + Inf.sFamily + "-" +
							Inf.sWeight + "-" + Inf.sSlant + "-" + Inf.sSetwidth + "--" + 
							hdr.stFontBBX.nBBh.ToString() + "-" +
							(hdr.stSize.nPoint * 10).ToString() + "-75-75-" +
							Inf.sSpacing + "-" + (hdr.stFontBBX.nBBw * 10).ToString() + "-" +
							Inf.sCharset + "-1";
				}
				file.WriteLine(line);
				
				//フォントのサイズ情報を出力
				line = "SIZE " + hdr.stSize.nPoint.ToString() + " " + 
						hdr.stSize.nXres.ToString() + " " + hdr.stSize.nYres.ToString();
				file.WriteLine(line);
	
				//フォント全体のバウンディングボックス情報を出力
				line = "FONTBOUNDINGBOX " + hdr.stFontBBX.nBBw.ToString() + " " + 
											hdr.stFontBBX.nBBh.ToString() + " " +
											hdr.stFontBBX.nXoff.ToString() + " " +
											hdr.stFontBBX.nYoff.ToString();
				file.WriteLine(line);

				//プロパティ部開始とプロパティ数を出力
				line = "STARTPROPERTIES " + m_BdfHdr.slProperty.Count.ToString();
				file.WriteLine(line);

				//プロパティ部を出力
				for (i = 0; i < m_BdfHdr.slProperty.Count; i++) {
					file.WriteLine(m_BdfHdr.slProperty[i]);
				}
				//プロパティ部終了を出力
				file.WriteLine("ENDPROPERTIES");
				
				//文字情報開始と文字数を出力
				line = "CHARS " + count.ToString();
				file.WriteLine(line);
			} catch {
				res = false;
			}
			return res;
		}

		//BDFフォントの文字ヘッダ部を保存する
		public bool SaveBdfCharHead(StreamWriter file, int idx)
		{
			bool	res = true;
            string	line = "";

			try {
				//文字開始と文字の名前を出力
				line = "STARTCHAR " + m_CharHdr[idx].sStartChar;
				file.WriteLine(line);

				//エンコード番号を出力
				line = "ENCODING " + m_CharHdr[idx].nEncode.ToString();
				file.WriteLine(line);

				//スケーリングされた次の文字の位置情報を出力
				line = "SWIDTH " + m_CharHdr[idx].nSWidth_x.ToString() +
						" " + m_CharHdr[idx].nSWidth_y.ToString();
				file.WriteLine(line);

				//ピクセル単位の次の文字の位置情報を出力
				line = "DWIDTH " + m_CharHdr[idx].nDWidth_x.ToString() +
						" " + m_CharHdr[idx].nDWidth_y.ToString();
				file.WriteLine(line);

				//文字のバウンディングボックス情報を出力
				line = "BBX " + m_CharHdr[idx].stBBX.nBBw.ToString() +
						" " + m_CharHdr[idx].stBBX.nBBh.ToString() +
						" " + m_CharHdr[idx].stBBX.nXoff.ToString() +
						" " + m_CharHdr[idx].stBBX.nYoff.ToString();
				file.WriteLine(line);
			} catch {
				res = false;
			}
			return res;
		}

		//BDFフォントの文字のビットマップ部を保存する
		public bool SaveBdfCharBitmap(StreamWriter file, int idx)
		{
			bool	res = true;
            string	line = "";
			int		xoff, yoff;
			int		w, h;
			int		line_bytes;
			int		i, j, n;
			byte[]	buf;		
			byte[]	fnt = new byte[] { 0x00 };

			//バウンディングボックスから縦横ピクセル数とラインのバイト数を得る
			w = m_CharHdr[idx].stBBX.nBBw;
			h = m_CharHdr[idx].stBBX.nBBh;
			line_bytes = w / 8;
			if (w % 8 > 0) {
				line_bytes++;
			}
			//全体のバウンディングボックスと文字のバウンディングボックスからオフセットを計算
			xoff = m_CharHdr[idx].stBBX.nXoff - m_BdfHdr.stFontBBX.nXoff;
			yoff = m_nHeight - (h + (m_CharHdr[idx].stBBX.nYoff - m_BdfHdr.stFontBBX.nYoff));

			//フォントのビットマップデータを取り出す
			buf = GetFontData(idx);		

			if (xoff > 0) {
				//オフセットが＋ならビットマップを左にシフト
				for (i = 0; i < xoff; i++) {
					ShiftLeft(m_nWidth, m_nHeight, m_nLineBytes, ref buf);
				}
			} else if (xoff < 0) {
				//オフセットが－ならビットマップを右にシフト
				for (i = 0; i > xoff; i--) {
					ShiftRight(m_nWidth, m_nHeight, m_nLineBytes, ref buf);
				}
			}
			if (yoff > 0) {
				//オフセットが＋ならビットマップを上にシフト
				for (i = 0; i < yoff; i++) {
					ShiftUp(m_nWidth, m_nHeight, m_nLineBytes, ref buf);
				}
			} else if (yoff < 0) {
				//オフセットが－ならビットマップを下にシフト
				for (i = 0; i > yoff; i--) {
					ShiftDown(m_nWidth, m_nHeight, m_nLineBytes, ref buf);
				}
			}
			try {
				//ビットマップ部の開始を出力
				file.WriteLine("BITMAP");

				//１ラインを１行のデータとして出力する
				for (i = 0; i < h; i++) {
					//行データクリア
					line = "";

					//ライン先頭のバイト位置
					n = i * m_nLineBytes;

					//1ラインのバイト数だけ繰り返す
					for (j = 0; j < line_bytes; j++) {
						//1バイトを2文字の16進数文字列で行データに追加
						fnt[0] = buf[n + j];
						line += BitConverter.ToString(fnt);
					}
					//１行のデータを出力
					file.WriteLine(line);
				}
				//文字データ終了を出力
				file.WriteLine("ENDCHAR");
			} catch {
				res = false;
			}
			return res;
		}


		//フォントファイルをBDF形式で保存する
		public bool SaveBdf(String fname, FontInfo Inf, int start = 0, int end = -1)
		{
			if (!m_bEnable) {
				return false;
			}
			StreamWriter	file;
			int				fnt_num;
			int				idx;
			bool			res = true;

			fnt_num = m_lstFontList.Count;

			if (end < 0) {
				end = fnt_num - 1;
			}
			if (end < start) {
				return false;
			}
			fnt_num = end - start + 1;

			try {
				//ファイルを書き込みオープン
	            file = new StreamWriter(fname, false);

				//BDFフォントファイルのヘッダ部を保存する
				if (!SaveBdfHead(file, m_BdfHdr, Inf, fnt_num)) {
					file.Close();
					return false;
				}
				for (idx = start; idx <= end; idx++) {
					//BDFフォントの文字ヘッダ部を保存する
					if (!(res = SaveBdfCharHead(file, idx))) {
						break;
					}
					//BDFフォントの文字のビットマップ部を保存する
					if (!(res = SaveBdfCharBitmap(file, idx))) {
						break;
					}
				}
				if (res) {
					//フォント終了を出力
					file.WriteLine("ENDFONT");
				}
				file.Close();

				m_FontInf = Inf;
			} catch {
				res = false;
			}
			return res;
		}

		//フォントの文字データを取得する
		public	byte[] GetFontData(int idx)
		{
			if (!m_bEnable) {
				return null;
			}
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return null;
			}
			return (byte[])m_lstFontList[idx].Clone();
		}

		//フォントの文字データをクリアする
		public bool ClearFontData(int idx)
		{
			if (!m_bEnable) {
				return false;
			}
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			ClearFontDataBytes(m_lstFontList[idx]);

			return true;
		}

		//フォントの文字データをセット
		public bool	SetFontData(int idx, byte[] data)
		{
			if (!m_bEnable) {
				return false;
			}
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			data.CopyTo(m_lstFontList[idx], 0);

			return true;
		}

		//フォントの文字データ名をセットする
		public String GetFontDataName(int idx)
		{
			if (!m_bEnable) {
				return "";
			}
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return "";
			}
			return m_CharHdr[idx].sStartChar;
		}

		//フォントの文字データ名をセットする
		public bool SetFontDataName(int idx, String name)
		{
			if (!m_bEnable) {
				return false;
			}
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			m_CharHdr[idx].sStartChar = name;

			return true;
		}

		//フォントの文字データのヘッダを初期化する
		public void	InitCharHdr(int idx, String name, ref CharHead hdr)
		{
			double	param;
			String	glyf_name;

			if (name.Length > 0) {
				glyf_name = name;
			} else {
				glyf_name = idx.ToString();
			}
			param = (double)m_nWidth / ((double)PIXEL_PER_INCH / (double)POINT_PER_INCH) * (double)1000;

			hdr.sStartChar = glyf_name;
			hdr.nEncode = idx;
			hdr.stBBX = m_BdfHdr.stFontBBX;
			hdr.nDWidth_x = m_BdfHdr.stFontBBX.nBBw;
			hdr.nDWidth_y = 0;
			hdr.nSWidth_x = (int)Math.Ceiling(param / m_BdfHdr.stSize.nPoint);
			hdr.nSWidth_y = 0;
		}

		//フォントの文字データを追加
		public bool	AddFontData(byte[] data, CharHead hdr)
		{
			if (!m_bEnable) {
				return false;
			}
			int	idx = m_lstFontList.Count;
			
			idx = m_CharHdr.Count;

			if (data == null) {
				m_lstFontList.Add(new byte[m_nDataSize]);
				ClearFontDataBytes(m_lstFontList[idx]);
			} else {
				m_lstFontList.Add((byte[])data.Clone());
			}
			m_CharHdr.Add(new CharHead(hdr));

			m_BdfHdr.nChars++;

			return true;
		}

		//フォントの文字データを追加
		public bool	AddFontData(byte[] data, String name = "")
		{
			if (!m_bEnable) {
				return false;
			}
			bool		res;
			CharHead	hdr = new CharHead();
			int			idx = m_lstFontList.Count;

			InitCharHdr(idx, name, ref hdr);

			res = AddFontData(data, hdr);

			hdr = null;

			return res;
		}

		//フォントの文字データを挿入
		public bool	InsertFontData(int idx, byte[] data, String name = "")
		{
			if (!m_bEnable) {
				return false;
			}
			CharHead	hdr = new CharHead();

			InitCharHdr(idx, name, ref hdr);

			if (idx < 0) {
				return false;
			} else if (idx < m_lstFontList.Count) {
				if (data == null) {
					m_lstFontList.Insert(idx, new byte[m_nDataSize]);
					ClearFontDataBytes(m_lstFontList[idx]);
				} else {
					m_lstFontList.Insert(idx, (byte[])data.Clone());
				}
				m_CharHdr.Insert(idx, new CharHead(hdr));
			} else if (idx == m_lstFontList.Count) {
				if (data == null) {
					m_lstFontList.Add(new byte[m_nDataSize]);
					ClearFontDataBytes(m_lstFontList[idx]);
				} else {
					m_lstFontList.Add((byte[])data.Clone());
				}
				m_CharHdr.Add(new CharHead(hdr));
			} else {
				return false;
			}
			m_BdfHdr.nChars++;

			return true;
		}

		//フォントの文字データを削除
		public bool DeleteFontData(int idx)
		{
			if (!m_bEnable) {
				return false;
			}
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			m_lstFontList[idx] = null;
			m_lstFontList.RemoveAt(idx);

			m_CharHdr.RemoveAt(idx);

			m_BdfHdr.nChars--;

			return true;
		}

		//フォントデータを右にシフトする
		public static void ShiftRight(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		x, y, pos;
			byte	f, old_f;
			
			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			for (y = 0; y < height; y++) {
				f = 0;
				old_f = 0;
				for (x = 0; x < line_bytes; x++) {
					pos = y * line_bytes + x;
					f = (byte)(buf[pos] & 0x01);
					buf[pos] >>= 1;
					if (old_f != 0) {
						buf[pos] |= 0x80;
					} else {
						buf[pos] &= 0x7F;
					}
					old_f = f;
				}
			}
		}
		//フォントデータを左にシフトする
		public static void ShiftLeft(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		x, y, pos;
			byte	f, old_f;
			
			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			for (y = 0; y < height; y++) {
				f = 0;
				old_f = 0;
				for (x = line_bytes - 1; x >= 0; x--) {
					pos = y * line_bytes + x;
					f = (byte)(buf[pos] & 0x80);
					buf[pos] <<= 1;
					if (old_f != 0) {
						buf[pos] |= 0x01;
					} else {
						buf[pos] &= 0xfe;
					}
					old_f = f;
				}
			}
		}
		//フォントデータを上にシフトする
		public static void ShiftUp(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		x, y, spos, dpos;

			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			for (y = 0; y < (height - 1); y++) {
				dpos = y * line_bytes;
				spos = dpos + line_bytes;
				for (x = 0; x < line_bytes; x++) {
					buf[dpos++] = buf[spos++];
				}
			}
			dpos = y * line_bytes;
			for (x = 0; x < line_bytes; x++) {
				buf[dpos++] = 0;
			}
		}

		//フォントデータを下にシフトする
		public static void ShiftDown(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		x, y, spos, dpos;

			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			for (y = height - 1; y > 0; y--) {
				dpos = y * line_bytes;
				spos = dpos - line_bytes;
				for (x = 0; x < line_bytes; x++) {
					buf[dpos++] = buf[spos++];
				}
			}
			dpos = y * line_bytes;
			for (x = 0; x < line_bytes; x++) {
				buf[dpos++] = 0;
			}
		}

		//フォントデータを左右反転する
		public static void 	FlipHorizontal(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		x, y, spos, dpos;
			int		sx, dx;
			byte	smask, dmask;

			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			byte[]	tmp = new byte[line_bytes];

			for (y = 0; y < height; y++) {
				Array.Clear(tmp, 0, line_bytes);
				for (sx = width - 1; sx >= 0; sx--) {
					dx = (width - 1) - sx;
					spos = y * line_bytes + (sx / 8);
					dpos = dx / 8;
					smask = (byte)(0x80 >> (sx % 8));
					dmask = (byte)(0x80 >> (dx % 8));
					if ((buf[spos] & smask) != 0x00) {
						tmp[dpos] |= dmask;
					}
				}
				dpos = y * line_bytes;
				for (x = 0; x < line_bytes; x++) {
					buf[dpos + x] = tmp[x];
				}
			}
		}

		//フォントデータを上下反転する
		public static void 	FlipVertical(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		pos, spos, dpos;
			int		sy, dy;
			byte	tmp;

			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			for (sy = 0; sy < (height / 2); sy++) {
				dy = (height - 1) - sy;
				for (pos = 0; pos < line_bytes; pos++) {
					spos = sy * line_bytes + pos;
					dpos = dy * line_bytes + pos;
					tmp = buf[spos];
					buf[spos] = buf[dpos];
					buf[dpos] = tmp;
				}
			}
		}

		//フォントデータをネガポジ反転する
		public static void 	Reverse(int width, int height, int line_bytes, ref byte[] buf)
		{
			int		x, y, pos;
			byte	mask;

			if (line_bytes <= 0) {
				line_bytes = width / 8;
				if (width % 8 > 0) {
					line_bytes++;
				}
			}
			for (y = 0; y < height; y++) {
				mask = 0x80;
				for (x = 0; x < width; x++) {
					mask = (byte)(0x80 >> (x % 8));
					pos = y * line_bytes + (x / 8);
					if ((buf[pos] & mask) != 0x00) {
						buf[pos] &= (byte)~mask;
					} else {
						buf[pos] |= mask;
					}
				}
			}
		}

		//フォントデータを右にシフトする
		public bool ShiftRight(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			ShiftRight(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータを左にシフトする
		public bool ShiftLeft(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			ShiftLeft(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータを上にシフトする
		public bool ShiftUp(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			ShiftUp(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータを下にシフトする
		public bool ShiftDown(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			ShiftDown(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータを左右反転する
		public bool FlipHorizontal(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			FlipHorizontal(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータを上下反転する
		public bool FlipVertical(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			FlipVertical(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータをネガポジ反転する
		public bool Reverse(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return false;
			}
			byte[]	buf = GetFontData(idx);

			Reverse(m_nWidth, m_nHeight, m_nLineBytes, ref buf);

			return SetFontData(idx, buf);
		}

		//フォントデータをテキスト形式でクリップボードにコピーする
		public void CopyToClipboard(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx) {
				return;
			}
			string	line;
			string item;
			int i, j;
			string buf = "";
			byte[] fnt = new byte[] { 0x00 };

			line = "";
			j = 0;
			for (i = 0; i < m_nDataSize; i++) {
				fnt[0] = m_lstFontList[idx][i];
				item = "0x" + BitConverter.ToString(fnt);
				if (i < m_nDataSize - 1) {
					item += ",";
				}
				line += item;
				j++;
				if (j >= m_nLineBytes) {
					buf += line + "\n";
					j = 0;
					line = "";
				}
			}
			if (j > 0) {
				buf += line;
			}
			Clipboard.SetDataObject(buf, true);
		}

		//テキスト形式のフォントデータをクリップボードから取り込む
		public void PasteFromClipboard(int idx)
		{
			if (idx < 0 || m_lstFontList.Count <= idx)
			{
				return;
			}
			IDataObject data = Clipboard.GetDataObject();
			if (!data.GetDataPresent(DataFormats.Text))
			{
				return;
			}
			string		buf = (string)data.GetData(DataFormats.Text);
			if (buf.Length <= 0)
			{
				return;
			}
			string[]	lines;
			string[]	items;
			int			n;
			int			i, j, pos;
			string		str;

			ClearFontData(idx);

			lines = buf.Split('\n');

			n = 0;
			for (i = 0; i < lines.Length; i++) {
				//行データをカンマで分割
				items = lines[i].Split(',');

				for (j = 0; j < items.Length && n < m_nDataSize; j++) {
					//各バイトの先頭は0xが付いている
					pos = items[j].IndexOf("0x");
					if (pos >= 0) {
						//先頭の0xを取り除く
						str = items[j].Substring(pos + 2, 2);
						if (str.Length >= 2) {
							//16進数文字列をバイトデータに変換して格納する
							((byte[])m_lstFontList[idx])[n++] = (byte)Convert.ToInt16(str, 16);
						}
					}
				}
			}
		}

		//フォントの文字データをクリアする
		private void	ClearFontDataBytes(byte[] data)
		{
			int	i;

			for (i = 0; i < m_nDataSize; i++) {
				data[i] = 0x00;
			}
		}
	}
}
