﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Reflection;
using System.Diagnostics;

namespace FontEdit
{
    enum eFileType
    {
        FILE_TYPE_NONE = 0,
        FILE_TYPE_C_ARRAY,
        FILE_TYPE_BDF,
		FILE_TYPE_NUM
    };

    public partial class FormMain : Form
    {
        const int MAX_FONT_WIDTH = 80;
        const int MAX_FONT_HEIGHT = 80;
        const int FFLT_FONT_WIDTH = 24;
        const int DFLT_FONT_HEIGHT = 24;
        const int DFLT_LINE_BYTES = 3;

        const int FILE_TYPE_ARRAY = 0;
        const int FILE_TYPE_BDF = 1;

        Size m_PictSize;
        int m_WPitch;
        int m_HPitch;

        string m_FormTitle;
        string m_FileName;


        BBox m_FontBBox;
        int m_nLineBytes;

        FontInfo m_FontInf;

        byte[] m_FontPtn = null;

        FontData m_FontData = new FontData();
        int m_nFontIdx = -1;

        Point m_Pos;
        bool m_bPress;
        CIniFile m_IniFile;

        eFileType m_eFileType;
		int m_nLastFilterIdx;

        //コンストラクタ
        public FormMain()
        {
            InitializeComponent();

            m_FontInf.sFoundry = "Misc";
            m_FontInf.sFamily = "Fixed";
            m_FontInf.sWeight = "Medium";
            m_FontInf.sSlant = "R";
            m_FontInf.sSetwidth = "Normal";
            m_FontInf.sSpacing = "M";
            m_FontInf.sCharset = "ISO8859";

            m_FontBBox.nBBw = FFLT_FONT_WIDTH;
            m_FontBBox.nBBh = DFLT_FONT_HEIGHT;
            m_FontBBox.nXoff = 0;
            m_FontBBox.nYoff = 0;

            m_nLineBytes = DFLT_LINE_BYTES;

            m_Pos.X = -1;
            m_Pos.Y = -1;
            m_bPress = false;

			m_eFileType = eFileType.FILE_TYPE_NONE;
			m_nLastFilterIdx = 1;
        }

        //フォーム・ロード時の処理
        private void FormMain_Load(object sender, EventArgs e)
        {
            panelCfg.Anchor = (AnchorStyles.Right | AnchorStyles.Top | AnchorStyles.Left);

            PictBox.Anchor = (AnchorStyles.Bottom | AnchorStyles.Right | AnchorStyles.Top | AnchorStyles.Left);

            m_FormTitle = Application.ProductName;

            m_FileName = "";

            //INIファイル読み込み
            LoadIniFile();

            CalcDotPitch();

            //フォントデータを生成
            m_FontData.Create(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes);

            m_FontBBox = m_FontData.GetFontBBox();

            m_FontData.AddFontData(null, "");

            SetFontIndex(0);
        }

        //フォーム・クローズ時の処理
        private void FormMain_Closing(object sender, FormClosingEventArgs e)
        {
            SaveIniFile();

            m_FontData.Delete();
        }

        //描画処理
        private void PictBox_Paint(object sender, PaintEventArgs e)
        {
            Graphics graph = e.Graphics;
            Pen rect_pen = new Pen(Color.Black);
            Pen line_pen = new Pen(Color.Black);
            Pen offset_pen = new Pen(Color.Red);
            int i, j;
            int row, col;
            int x1, y1, x2, y2;
            int idx, bits;
            int xoff, yoff;

            xoff = m_FontBBox.nXoff * -1;
            yoff = m_FontBBox.nBBh + m_FontBBox.nYoff;

            rect_pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Solid;
            line_pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
            offset_pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;

            if (m_FontPtn != null)
            {
                for (row = 0; row < m_FontBBox.nBBh; row++)
                {
                    y1 = m_HPitch * row;
                    y2 = y1 + m_HPitch;

                    for (i = 0; i < m_nLineBytes; i++)
                    {
                        idx = (row * m_nLineBytes) + i;
                        bits = m_FontBBox.nBBw - (i * 8);
                        if (bits > 8)
                        {
                            bits = 8;
                        }
                        for (j = 0; j < bits; j++)
                        {
                            col = i * 8 + j;
                            x1 = m_WPitch * col;
                            x2 = x1 + m_WPitch;

                            if ((m_FontPtn[idx] & (0x80 >> j)) != 0)
                            {
                                graph.FillRectangle(Brushes.Black, x1, y1, x2 - x1, y2 - y1);
                            }
                        }
                    }
                }
            }
            //graph.DrawRectangle(rect_pen, 0, 0, m_WPitch * m_FontBBox.nBBw, m_HPitch * m_FontBBox.nBBh);

            for (i = 0; i <= m_FontBBox.nBBh; i++)
            {
                x1 = 0;
                y1 = m_HPitch * i;
                x2 = m_WPitch * m_FontBBox.nBBw;
                y2 = y1;

                if (i == yoff)
                {
                    graph.DrawLine(offset_pen, x1, y1, x2, y2);
                }
                else
                {
                    graph.DrawLine(line_pen, x1, y1, x2, y2);
                }
            }
            for (i = 0; i <= m_FontBBox.nBBw; i++)
            {
                x1 = m_WPitch * i;
                y1 = 0;
                x2 = x1;
                y2 = m_HPitch * m_FontBBox.nBBh;
                if (i == xoff)
                {
                    graph.DrawLine(offset_pen, x1, y1, x2, y2);
                }
                else
                {
                    graph.DrawLine(line_pen, x1, y1, x2, y2);
                }
            }
        }

        //フォーム・サイズ変更処理
        private void FormMain_Resize(object sender, EventArgs e)
        {
            CalcDotPitch();

            PictBox.Invalidate();
        }

        //描画領域でのマウスボタンダウン処理
        private void PictBox_MouseDown(object sender, MouseEventArgs e)
        {
            Point pos;

            pos = new Point(GetCol(e.X), GetRow(e.Y));

            if (pos.X < 0 || m_FontBBox.nBBw <= pos.X ||
                pos.Y < 0 || m_FontBBox.nBBh <= pos.Y)
            {
                return;
            }
            m_Pos = pos;

            if (e.Button == MouseButtons.Left)
            {
                SetDot(m_Pos.Y, m_Pos.X);
            }
            else if (e.Button == MouseButtons.Right)
            {
                ClearDot(m_Pos.Y, m_Pos.X);
            }
            m_bPress = true;
        }

		//描画領域でのマウスボタンアップ処理
		private void PictBox_MouseUp(object sender, MouseEventArgs e)
        {
			m_Pos.X = -1;
			m_Pos.Y = -1;

			m_bPress = false;
        }

		//描画領域でのマウス移動処理
        private void PictBox_MouseMove(object sender, MouseEventArgs e)
        {
			if (m_bPress)
			{
				Point pos;

				pos = new Point(GetCol(e.X), GetRow(e.Y));
				if (pos.X < 0 || m_FontBBox.nBBw <= pos.X ||
						pos.Y < 0 || m_FontBBox.nBBh <= pos.Y)
				{
					return;
				}
				if (pos != m_Pos)
				{
					m_Pos = pos;
					if (e.Button == MouseButtons.Left)
					{
						SetDot(m_Pos.Y, m_Pos.X);
					}
					else if (e.Button == MouseButtons.Right)
					{
						ClearDot(m_Pos.Y, m_Pos.X);
					}
				}
			}
		}

        //ファイルメニューのロード処理
        private void MenuFileLoad_Click(object sender, EventArgs e)
        {
			LoadFile();
        }

        //ファイルメニューの保存処理
        private void MenuFileSave_Click(object sender, EventArgs e)
        {
			SaveFile();
        }

        //ファイルメニューの名前を付けて保存処理
        private void MenuFileSaveAs_Click(object sender, EventArgs e)
        {
            SaveFileAs();
        }

		//ファイルメニューの範囲を指定して保存
		private void MenuFileSaveRange_Click(object sender, EventArgs e)
		{
			SaveRange();
		}

        //ファイルメニューの全クリア処理
        private void MenuFileClear_Click(object sender, EventArgs e)
        {
            m_FileName = "";

            m_FontData.Delete();
            m_FontData.Create(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes);
            m_FontBBox = m_FontData.GetFontBBox();

            m_FontData.AddFontData(null, "");

            this.Text = m_FormTitle;

            SetFontIndex(0);
        }

        //ファイルメニューの終了処理
        private void MenuFileExit_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        //表示メニュー・前へ(Prev)の処理
        private void MenuViewPrev_Click(object sender, EventArgs e)
        {
            Prev();
        }

        //表示メニュー・次へ(Next)の処理
        private void MenuViewNext_Click(object sender, EventArgs e)
        {
            Next();
        }

        //編集メニュー・追加(Add)の処理
        private void MenuEditAdd_Click(object sender, EventArgs e)
        {
            Add();
        }

        //編集メニュー・削除(Delete)の処理
        private void MenuEditDelete_Click(object sender, EventArgs e)
        {
            Delete();
        }

        //編集メニュー・挿入(Insert)の処理
        private void MenuEditInsert_Click(object sender, EventArgs e)
        {
            Insert();
        }

		//編集メニュー・範囲削除(Delete Range)の処理
		private void MenuEditDeleteRange_Click(object sender, EventArgs e)
		{
			DeleteRange();
		}

        //編集メニュー・上シフト(Up)の処理
        private void MenuEditUp_Click(object sender, EventArgs e)
        {
            ShiftUp();
        }

        //編集メニュー・下シフト(Down)の処理
        private void MenuEditDown_Click(object sender, EventArgs e)
        {
            ShiftDown();
        }

        //編集メニュー・左シフト(Left)の処理
        private void MenuEditLeft_Click(object sender, EventArgs e)
        {
            ShiftLeft();
        }

        //編集メニュー・右シフト(Right)の処理
        private void MenuEditRight_Click(object sender, EventArgs e)
        {
			ShiftRight();
        }

        //編集メニュー・左右反転(Flip Horizontal)の処理
		private void MenuEditFlipHorizontal_Click(object sender, EventArgs e)
		{
			FlipHorizontal();
		}

        //編集メニュー・上下反転(Flip Vertical)の処理
		private void MenuEditFlipVertical_Click(object sender, EventArgs e)
		{
			FlipVertical();
		}

        //編集メニュー・ネガポジ反転(Reverse)の処理
		private void MenuEditReverse_Click(object sender, EventArgs e)
		{
			Reverse();
		}

        //編集メニュー・編集破棄(Reset)の処理
        private void MenuEditReset_Click(object sender, EventArgs e)
        {
            EditReset();
        }

		//編集メニュー・コピー
		private void MenuEditCopy_Click(object sender, EventArgs e)
		{
			if (0 <= m_nFontIdx && m_nFontIdx < m_FontData.Count())
			{
				m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

				m_FontData.CopyToClipboard(m_nFontIdx);
			}
		}

		//編集メニュー・貼り付け
		private void MenuEditPaste_Click(object sender, EventArgs e)
		{
			if (0 <= m_nFontIdx && m_nFontIdx < m_FontData.Count())
			{
				m_FontData.PasteFromClipboard(m_nFontIdx);

				SetFontIndex(m_nFontIdx);
			}
		}

        //オプションメニュー・FontDataの処理
        private void MenuOptionFontData_Click(object sender, EventArgs e)
        {
            FormFontInfo dlg = new FormFontInfo();

            dlg.m_FontInf = m_FontInf;

            dlg.m_nWidth = m_FontBBox.nBBw;
            dlg.m_nHeight = m_FontBBox.nBBh;
            dlg.m_nXoffset = m_FontBBox.nXoff;
            dlg.m_nYoffset = m_FontBBox.nYoff * -1;
            dlg.m_nLineBytes = m_nLineBytes;

            if (dlg.ShowDialog(this) == DialogResult.OK)
            {
                int width;
                int height;
                int xoff;
                int yoff;
                int line_bytes;

                width = dlg.m_nWidth;
                height = dlg.m_nHeight;
                line_bytes = dlg.m_nLineBytes;
                xoff = dlg.m_nXoffset;
                yoff = dlg.m_nYoffset * -1;

                m_FontInf = dlg.m_FontInf;

                if (width != m_FontBBox.nBBw || height != m_FontBBox.nBBh || line_bytes != m_nLineBytes ||
                                                    xoff != m_FontBBox.nXoff || yoff != m_FontBBox.nYoff)
                {
                    //サイズ情報を変更されたので読み込み済みのデータはすべて無効となる
                    m_FileName = "";

                    m_FontBBox.nBBw = width;
                    m_FontBBox.nBBh = height;
                    m_FontBBox.nXoff = xoff;
                    m_FontBBox.nYoff = yoff;
                    m_nLineBytes = line_bytes;

                    m_FontData.Delete();
                    m_FontData.Create(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes);

                    m_FontBBox = m_FontData.GetFontBBox();

                    m_FontData.AddFontData(null, "");

                    m_nFontIdx = 0;
                }
                m_WPitch = (m_PictSize.Width - 1) / m_FontBBox.nBBw;
                m_HPitch = (m_PictSize.Height - 1) / m_FontBBox.nBBh;

                SetFontIndex(m_nFontIdx);
            }
            dlg.Dispose();
        }

        //ヘルプメニュー・バージョン(Version)の処理
        private void MenuHelpVersion_Click(object sender, EventArgs e)
        {
            ShowVersion();
        }

        //ヘルプメニュー・Readme JP の処理
		private void MenuHelpReadme_Click(object sender, EventArgs e)
		{
			string	strPath = System.Windows.Forms.Application.ExecutablePath;
			string	strDir = System.IO.Path.GetDirectoryName(strPath);
			string	strFileName = strDir + "\\" + "Readme_jp.txt";

			if (!File.Exists(strFileName)) {
				MessageBox.Show(strFileName + "\nFile not found", "Error");
				return;
			}
			System.Diagnostics.Process proc =
						System.Diagnostics.Process.Start(strFileName);
		}

        //ヘルプメニュー・Readme Enの処理
		private void MenuHelpReadmeEn_Click(object sender, EventArgs e)
		{
			string	strPath = System.Windows.Forms.Application.ExecutablePath;
			string	strDir = System.IO.Path.GetDirectoryName(strPath);
			string	strFileName = strDir + "\\" + "Readme_en.txt";

			if (!File.Exists(strFileName)) {
				MessageBox.Show(strFileName + "\nFile not found", "Error");
				return;
			}
			System.Diagnostics.Process proc =
						System.Diagnostics.Process.Start(strFileName);
		}

		//前(<<Prev)へボタンの処理
        private void btnPrev_Click(object sender, EventArgs e)
        {
			Prev();
        }

		//次へ(Next>>)ボタンの処理
        private void btnNext_Click(object sender, EventArgs e)
        {
			Next();
        }

		//追加(Add)ボタンの処理
        private void btnAdd_Click(object sender, EventArgs e)
        {
			Add();
        }

		//削除(Delete)ボタンの処理
        private void btnDelete_Click(object sender, EventArgs e)
        {
			Delete();
        }

		//挿入(Insert)ボタンの処理
        private void btnInsert_Click(object sender, EventArgs e)
        {
			Insert();
        }

		//上シフト(Up)ボタンの処理
        private void btnUp_Click(object sender, EventArgs e)
        {
			ShiftUp();
        }

		//下シフト(Down)ボタンの処理
        private void btnDown_Click(object sender, EventArgs e)
        {
			ShiftDown();
        }

		//左シフト(Left)ボタンの処理
        private void btnLeft_Click(object sender, EventArgs e)
        {
			ShiftLeft();
        }

		//右シフト(Right)ボタンの処理
        private void btnRight_Click(object sender, EventArgs e)
        {
			ShiftRight();
        }

		//左右反転(Flip Hrz)ボタンの処理
		private void btnFlipHrz_Click(object sender, EventArgs e)
		{
			FlipHorizontal();
		}

		//上下反転(Flip Vrt)ボタンの処理
		private void btnFlipVrt_Click(object sender, EventArgs e)
		{
			FlipVertical();
		}

		//ネガポジ反転(Rev)ボタンの処理
		private void btnReverse_Click(object sender, EventArgs e)
		{
			Reverse();
		}

		//編集破棄(Reset)ボタンの処理
        private void btnReset_Click(object sender, EventArgs e)
        {
			EditReset();
        }

		//Indexテキストボックスのキーダウン処理
		private void textBoxIndex_KeyDown(object sender, KeyEventArgs e)
		{
			if (e.KeyCode == Keys.Return)
			{
				int idx = int.Parse(textBoxIndex.Text);
				if (0 <= idx && idx < m_FontData.Count())
				{
					SetFontIndex(idx);
				}
				else
				{
					textBoxIndex.Text = m_nFontIdx.ToString();
				}
			}
		}

        //バージョン情報表示
        private void ShowVersion()
        {
			string copyright = "";
			string version = "";
			string msg;
			FileVersionInfo ver = FileVersionInfo.GetVersionInfo(Assembly.GetExecutingAssembly().Location);
			object[] obj = Assembly.GetEntryAssembly().GetCustomAttributes(typeof(AssemblyCopyrightAttribute), false);

			if ((obj != null) && (obj.Length > 0))
            {
                copyright = ((AssemblyCopyrightAttribute)obj[0]).Copyright;
            }	
			version = ver.FileMajorPart.ToString() + "." + ver.FileMinorPart.ToString();

			msg = Application.ProductName + "Ver." + version;
            msg += "\n" + copyright;

			MessageBox.Show(msg, "About " + Application.ProductName);
        }

        //描画ピッチを計算する
        private void CalcDotPitch()
        {
            m_PictSize = PictBox.Size;

            m_WPitch = (m_PictSize.Width - 1) / m_FontBBox.nBBw;
            m_HPitch = (m_PictSize.Height - 1) / m_FontBBox.nBBh;
        }

        //INIファイル読み込み
        private void LoadIniFile()
        {
            string strPath;
            string strDir;
            string strAppName;
            string strFileName;
            int val;
            int min_val;

            strPath = System.Windows.Forms.Application.ExecutablePath;
            strDir = System.IO.Path.GetDirectoryName(strPath);

            strAppName = System.IO.Path.GetFileNameWithoutExtension(strPath);

            strFileName = strDir + "\\" + strAppName + ".ini";

            m_IniFile = new CIniFile(strFileName);

            val = m_IniFile.GetIntValue("FONT", "WIDTH", m_FontBBox.nBBw);
            if (0 < val && val <= MAX_FONT_WIDTH)
            {
                m_FontBBox.nBBw = val;
            }
            val = m_IniFile.GetIntValue("FONT", "HEIGHT", m_FontBBox.nBBh);
            if (0 < val && val <= MAX_FONT_HEIGHT)
            {
                m_FontBBox.nBBh = val;
            }
            min_val = m_FontBBox.nBBw / 8;
            if (m_FontBBox.nBBw % 8 > 0)
            {
                min_val++;
            }
            val = m_IniFile.GetIntValue("FONT", "LINE_BYTES", m_nLineBytes);
            if (val < min_val)
            {
                m_nLineBytes = min_val;
            }
            else
            {
                m_nLineBytes = val;
            }
            val = m_IniFile.GetIntValue("FONT", "XOFFSET", m_FontBBox.nXoff);
            if (0 < val && val <= MAX_FONT_HEIGHT)
            {
                m_FontBBox.nXoff = val;
            }
            val = m_IniFile.GetIntValue("FONT", "YOFFSET", m_FontBBox.nYoff * -1);
            if (m_FontBBox.nBBh - val > 0)
            {
                m_FontBBox.nYoff = val * -1;
            }
			val = m_IniFile.GetIntValue("FILE", "LAST_FILE_FILTER_INDEX", m_nLastFilterIdx);
			if (1 <= val && val <= 4)
			{
				m_nLastFilterIdx = val;
			}
		}

        //INIファイル保存
        private void SaveIniFile()
        {
            m_IniFile.SetIntValue("FONT", "WIDTH", m_FontBBox.nBBw);
            m_IniFile.SetIntValue("FONT", "HEIGHT", m_FontBBox.nBBh);
            m_IniFile.SetIntValue("FONT", "XOFFSET", m_FontBBox.nXoff);
            m_IniFile.SetIntValue("FONT", "YOFFSET", m_FontBBox.nYoff * -1);
            m_IniFile.SetIntValue("FONT", "LINE_BYTES", m_nLineBytes);
			m_IniFile.SetIntValue("FILE", "LAST_FILE_FILTER_INDEX", m_nLastFilterIdx);
		}

		//フォントファイルを読み込む
		private bool LoadFile()
		{
			OpenFileDialog dlg = new OpenFileDialog();

			dlg.Filter = "C/C++ Array Header File(*.h)|*.h|BDF Format File(*.bdf)|*.bdf|All Files(*.*)|*.*";
			dlg.FilterIndex = m_nLastFilterIdx;
			dlg.CheckFileExists = true;
			dlg.RestoreDirectory = true;
			dlg.Title = "Select the File for Loading";

			if (dlg.ShowDialog() == DialogResult.OK)
			{
				string ext;

				m_nLastFilterIdx = dlg.FilterIndex;

				ext = Path.GetExtension(dlg.FileName);

				if (ext.ToLower() == ".bdf")
				{
					if (LoadBdfFile(dlg.FileName))
					{
						this.Text = m_FormTitle + " " + dlg.FileName;
					}
				}
				else
				{
					if (LoadArrayFile(dlg.FileName))
					{
						this.Text = m_FormTitle + " " + dlg.FileName;
					}
				}
			}
			return true;
		}

		//フォントファイルを保存する
		private bool SaveFile()
		{
			bool	res = false;

            m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

            if (m_FileName.Length > 0)
            {
				if (m_eFileType == eFileType.FILE_TYPE_BDF)
                {
                    res = SaveBdfFile(m_FileName);
                }
                else
                {
                    res = SaveArrayFile(m_FileName);
                }
            }
            else
            {
                res = SaveFileAs();
            }
			return res;
		}

        //名前を付けてフォントデータに保存
        private bool SaveFileAs()
        {
            bool bRes = true;

            m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

            SaveFileDialog dlg = new SaveFileDialog();

			dlg.Filter = "C/C++ Array Header File(*.h)|*.h|BDF Format File(*.bdf)|*.bdf|All Files(*.*)|*.*";
			dlg.FilterIndex = m_nLastFilterIdx;
            dlg.OverwritePrompt = true;
            dlg.RestoreDirectory = true;
            dlg.CheckPathExists = true;
            dlg.Title = "Specify the File for Saving";

            if (dlg.ShowDialog() == DialogResult.OK) {
				string ext;

				m_nLastFilterIdx = dlg.FilterIndex;

				ext = Path.GetExtension(dlg.FileName);

				if (ext.ToLower() == ".bdf") {
					if (!(bRes = SaveBdfFile(dlg.FileName))) {
						MessageBox.Show("Save As " + dlg.FileName + " Error");
					}
				} else {
					if (!(bRes = SaveArrayFile(dlg.FileName))) {
						MessageBox.Show("Save As " + dlg.FileName + " Error");
					}
				}

            }
            return bRes;
        }
		//範囲を指定して保存
		private bool SaveRange()
		{
            bool	bRes = true;
			int		nStart = 0;
			int		nEnd = -1;

			m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

			FormRange range_dlg = new FormRange();

			range_dlg.m_sMsg = "Please input the Save Range.";

			range_dlg.m_nStartIdx = 0;
			range_dlg.m_nEndIdx = m_FontData.Count() - 1;

			if (range_dlg.ShowDialog(this) == DialogResult.OK) {
				nStart = range_dlg.m_nStartIdx;
				nEnd = range_dlg.m_nEndIdx;
				
				if (nStart > nEnd) {
					int tmp = nStart;

					nStart = nEnd;
					nEnd = tmp;
				}
				if (nStart >= m_FontData.Count()) {
					MessageBox.Show("There is no data in the specified range.");
					return false;
				}
				if (nEnd > m_FontData.Count() - 1) {
					nEnd = m_FontData.Count() - 1;
				}
				if (nEnd < nStart) {
					MessageBox.Show("There is no data to be saved.");
					return false;
				}
			} else {
				return false;
			}
            SaveFileDialog dlg = new SaveFileDialog();

			dlg.Filter = "C/C++ Array Header File(*.h)|*.h|BDF Format File(*.bdf)|*.bdf|All Files(*.*)|*.*";
			dlg.FilterIndex = m_nLastFilterIdx;
            dlg.OverwritePrompt = true;
            dlg.RestoreDirectory = true;
            dlg.CheckPathExists = true;
            dlg.Title = "Specify the File for Saving";

            if (dlg.ShowDialog() == DialogResult.OK) {
				string ext;

				m_nLastFilterIdx = dlg.FilterIndex;

				ext = Path.GetExtension(dlg.FileName);

				if (ext.ToLower() == ".bdf") {
					if (!(bRes = SaveBdfFile(dlg.FileName, nStart, nEnd))) {
						MessageBox.Show("Save As " + dlg.FileName + " Error");
					}
				} else {
					if (!(bRes = SaveArrayFile(dlg.FileName, nStart, nEnd))) {
						MessageBox.Show("Save As " + dlg.FileName + " Error");
					}
				}

            }
            return bRes;
		}

        //フォントファイル読み込み
        private bool LoadArrayFile(string fname)
        {
            if (!m_FontData.Load(fname, m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes))
            {
                return false;
            }
            m_FontBBox = m_FontData.GetFontBBox();
            m_nLineBytes = m_FontData.GetLineBytes();
            m_FontInf.sFoundry = "Misc";
            m_FontInf.sFamily = "Fixed";
            m_FontInf.sWeight = "Medium";
            m_FontInf.sSlant = "R";
            m_FontInf.sSetwidth = "Normal";
            m_FontInf.sSpacing = "M";
            m_FontInf.sCharset = "ISO8859";

            CalcDotPitch();

            if (m_FontData.Count() > 0)
            {
                m_FileName = fname;

                SetFontIndex(0);
            }
            else
            {
                m_nFontIdx = -1;
                textBoxCount.Text = m_FontData.Count().ToString();
            }
			m_eFileType = eFileType.FILE_TYPE_C_ARRAY;

            return true;
        }

        //BDFフォントファイル読み込み
        private bool LoadBdfFile(string fname)
        {
            if (!m_FontData.LoadBdf(fname))
            {
                return false;
            }
            m_FontBBox = m_FontData.GetFontBBox();
            m_nLineBytes = m_FontData.GetLineBytes();
            m_FontInf = m_FontData.GetFintInfo();

            CalcDotPitch();

            if (m_FontData.Count() > 0)
            {
                m_FileName = fname;

                SetFontIndex(0);
            }
            else
            {
                m_nFontIdx = -1;
                textBoxCount.Text = m_FontData.Count().ToString();
            }
			m_eFileType = eFileType.FILE_TYPE_BDF;

            return true;
        }

        //フォントファイル保存
        private bool SaveArrayFile(string fname, int start = 0, int end = -1)
        {
            if (!m_FontData.Save(fname, start, end))
            {
                return false;
            }
            return true;
        }

        //BDFファイルに保存
        private bool SaveBdfFile(string fname, int start = 0, int end = -1)
        {
            if (!m_FontData.SaveBdf(fname, m_FontInf, start, end))
            {
                return false;
            }
            return true;
        }

        //前(<<Prev)へボタンの処理
        private void Prev()
        {
            m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

            if (textBoxName.Text.Length > 0)
            {
                m_FontData.SetFontDataName(m_nFontIdx, textBoxName.Text);
            }
            if (m_nFontIdx > 0)
            {
                SetFontIndex(m_nFontIdx - 1);
            }
        }

        //次へ(Next>>)ボタンの処理
        private void Next()
        {
            m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

            if (textBoxName.Text.Length > 0)
            {
                m_FontData.SetFontDataName(m_nFontIdx, textBoxName.Text);
            }
            if (m_nFontIdx < m_FontData.Count() - 1)
            {
                SetFontIndex(m_nFontIdx + 1);
            }
        }

        //追加(Add)ボタンの処理
        private void Add()
        {
            m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

            if (textBoxName.Text.Length > 0)
            {
                m_FontData.SetFontDataName(m_nFontIdx, textBoxName.Text);
            }
            m_FontData.AddFontData(null, "");
            m_nFontIdx = m_FontData.Count() - 1;

            SetFontIndex(m_nFontIdx);
        }

        //削除(Delete)ボタンの処理
        private void Delete()
        {
            int num = m_FontData.Count();

            if (m_nFontIdx >= num)
            {
                return;
            }
            if (num > 1)
            {
                m_FontData.DeleteFontData(m_nFontIdx);
                num = m_FontData.Count();
                if (m_nFontIdx >= num)
                {
                    m_nFontIdx = num - 1;
                }
            }
            else
            {
                m_nFontIdx = 0;
                m_FontData.ClearFontData(m_nFontIdx);
            }
            SetFontIndex(m_nFontIdx);
        }

		//範囲を指定して削除
		private void DeleteRange()
		{
			FormRange dlg = new FormRange();

			dlg.m_sMsg = "Please input the Delete Range.";

			if (dlg.ShowDialog(this) == DialogResult.OK) {
				int nStart = dlg.m_nStartIdx;
				int nEnd = dlg.m_nEndIdx;
				int idx;
				
				if (nStart > nEnd) {
					int tmp = nStart;

					nStart = nEnd;
					nEnd = tmp;
				}
				if (nStart >= m_FontData.Count()) {
					return;
				}
				if (nEnd > m_FontData.Count() - 1) {
					nEnd = m_FontData.Count() - 1;
				}
				for (idx = nEnd; idx >= nStart; idx--) {
					if (0 <= idx && idx < m_FontData.Count()) {
						if (m_FontData.Count() > 1) {
							m_FontData.DeleteFontData(idx);
						} else {
							m_FontData.ClearFontData(idx);
							break;
						}
					}
				}
				if (m_nFontIdx > m_FontData.Count() - 1) {
					m_nFontIdx = m_FontData.Count() - 1;
				}
				SetFontIndex(m_nFontIdx);
			}
		}

		//挿入(Insert)ボタンの処理
		private void Insert()
        {
            m_FontData.SetFontData(m_nFontIdx, m_FontPtn);

            if (textBoxName.Text.Length > 0)
            {
                m_FontData.SetFontDataName(m_nFontIdx, textBoxName.Text);
            }
            m_FontData.InsertFontData(m_nFontIdx, null, "");

            SetFontIndex(m_nFontIdx);
        }

        //フォントを上にシフトする
        private void ShiftUp()
        {
            FontData.ShiftUp(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }

        //フォントを下にシフトする
        private void ShiftDown()
        {
            FontData.ShiftDown(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }

        //フォントを左にシフトする
        private void ShiftLeft()
        {
            FontData.ShiftLeft(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }

        //フォントを右にシフトする
        private void ShiftRight()
        {
            FontData.ShiftRight(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }
        //フォントを左右反転する
        private void FlipHorizontal()
        {
            FontData.FlipHorizontal(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }
        //フォントを上下反転する
        private void FlipVertical()
        {
            FontData.FlipVertical(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }
        //フォントをネガポジ反転する
        private void Reverse()
        {
            FontData.Reverse(m_FontBBox.nBBw, m_FontBBox.nBBh, m_nLineBytes, ref m_FontPtn);

            PictBox.Invalidate();
        }

        //編集内容を破棄する
        private void EditReset()
        {
            m_FontPtn = m_FontData.GetFontData(m_nFontIdx);

            PictBox.Invalidate();
        }

        //フォントのドットをセットする
        private void SetDot(int row, int col)
        {
            int idx;
            int bit_pos;
            byte bit_ptn;

            idx = GetDataIdx(row, col);
            bit_pos = GetDataBitPos(col);

            bit_ptn = (byte)(0x80 >> bit_pos);

            m_FontPtn[idx] |= bit_ptn;

            PictBox.Invalidate();
        }

        //フォントのドットをクリアする
        private void ClearDot(int row, int col)
        {
            int idx;
            int bit_pos;
            byte bit_ptn;

            idx = GetDataIdx(row, col);
            bit_pos = GetDataBitPos(col);

            bit_ptn = (byte)(0x80 >> bit_pos);

            m_FontPtn[idx] &= (byte)~bit_ptn;

            PictBox.Invalidate();
        }

        //マウスY座標のドットの行位置を得る
        private int GetRow(int y)
        {
            int row;

            row = y / m_HPitch;

            return row;
        }

        //マウスX座標のドットの列位置を得る
        private int GetCol(int x)
        {
            int col;

            col = x / m_WPitch;

            return col;
        }

        //ドットの行・列からフォントデータのバイト位置を得る
        private int GetDataIdx(int row, int col)
        {
            int idx;

            idx = (row * m_nLineBytes) + (col / 8);

            return idx;
        }

        //ドットの列からフォントデータ・バイト内のビット位置を得る
        private int GetDataBitPos(int col)
        {
            int pos;

            pos = col % 8;

            return pos;
        }

        //インデックスで指定されたフォントを表示する
        private void SetFontIndex(int Idx)
        {
            m_nFontIdx = Idx;
            m_FontPtn = m_FontData.GetFontData(m_nFontIdx);

            textBoxIndex.Text = m_nFontIdx.ToString();
            textBoxName.Text = m_FontData.GetFontDataName(m_nFontIdx);

            if (m_nFontIdx < m_FontData.Count() - 1)
            {
                btnNext.Enabled = true;
            }
            else
            {
                btnNext.Enabled = false;
            }
            if (m_nFontIdx > 0)
            {
                btnPrev.Enabled = true;
            }
            else
            {
                btnPrev.Enabled = false;
            }
            textBoxCount.Text = m_FontData.Count().ToString();

            PictBox.Invalidate();
        }
    }
}
