﻿/*
Copyright (c) 2014 Stephen Stair (sgstair@akkit.org)
Additional code Miguel Parra (miguelvp@msn.com)
Additional code by Franci Kapel: http://html-color-codes.info/Contact/
(Franci Kapel: A lot of my code was influenced by JadeW's work:  https://github.com/rzva/ThermalView)

Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
*/

using System;
using System.Drawing;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using System.IO;
using winusbdotnet.UsbDevices;
using AForge.Imaging.Filters;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;
using System.Globalization;
using System.Drawing.Drawing2D;
using System.Media;
using OpenCvSharp;
using OpenCvSharp.Extensions;

namespace SeekCamMon
{
    public partial class Form1 : Form
    {
        [DllImport("User32.dll", EntryPoint = "PostMessage")]
        public static extern Int32 PostMessage(IntPtr hWnd, UInt32 Msg, UInt32 wParam, UInt32 lParam);

        public const uint WM_USER = 0x400;
        public const uint WM_THREAD_MSG = WM_USER + 1;

        public const uint WP_THREAD_EXT_CAL = 1;

		//SeekThermal Compact Image Size (208x156)
        public const int	IMG_WIDTH = 208;
        public const int	IMG_HEIGHT = 156;
		//SeekThermal Compact Pro Image Size (320x240)
        public const int	PRO_IMG_WIDTH = 320;
        public const int	PRO_IMG_HEIGHT = 240;

		public const int	IMG_PIX_SIZE = IMG_WIDTH * IMG_HEIGHT;	//208*156=32448
		public const int	PRO_IMG_PIX_SIZE = PRO_IMG_WIDTH * PRO_IMG_HEIGHT;	//320*240=76800

		//SeekThermal Compact Image Cropped Size (206x156)
        public const int	IMG_CROPPED_WIDTH = 206;
        public const int	IMG_CROPPED_HEIGHT = 156;
		//SeekThermal Compact Pro Image Cropped Size (318x240)
        public const int	PRO_IMG_CROPPED_WIDTH = 318;
        public const int	PRO_IMG_CROPPED_HEIGHT = 240;

        public const int	IMG_WND_WIDTH = 412;
        public const int	IMG_WND_HEIGHT = 312;

        public const int	POINT_NUM = 5;
        public const int	PNT_VAL_NUM = IMG_WND_WIDTH;

		public const double	TEMP_RANGE_MAX = 330.0;
		public const double	TEMP_RANGE_MIN = -40.0;

        SeekThermal thermal;
        Thread thermalThread;
		ManualResetEvent m_pauseEvent;
		ThermalFrame currentFrame, lastUsableFrame;

        public const double fMIN_TEMP_LIMIT = -9999.9;
        public const double fMAX_TEMP_LIMIT = 9999.9;

        public const int	MIN_TREND_LIMIT = -9999;
        public const int	MAX_TREND_LIMIT = 9999;

        public const int	TREND_Y_MIN = -999;
        public const int	TREND_Y_MAX = 999;

        public const int MIN_POS_X = 0;
        public const int MIN_POS_Y = 0;
        public const int MAX_POS_X = 206;
        public const int MAX_POS_Y = 154;
        public const int PRO_MIN_POS_X = 0;
        public const int PRO_MIN_POS_Y = 0;
        public const int PRO_MAX_POS_X = 318;
        public const int PRO_MAX_POS_Y = 238;

        IntPtr m_hWnd;

        bool stopThread;
        bool grabExternalReference = false;
        bool usignExternalCal = false;
        bool saveImage = false;
        bool showMinVal = false;
        bool showMaxVal = false;
        bool autoRange = true;
        bool dynSliders = false;
        bool SharpenImage = false;
		bool m_isRunning = true;
		bool m_isRecording = false;
		
        string m_sModuleDir;
        string sIniFileName;
        string sAppName;
        string sProgVer;

		int	m_ImgWidth;
		int	m_ImgHeight;
		int m_ImgCropW;
		int m_ImgCropH;
		int	m_ImgPixSize;
		int	m_MinPosX;
		int	m_MinPosY;
		int	m_MaxPosX;
		int	m_MaxPosY;
		double	m_fImgScaleW;
		double	m_fImgScaleH;
		
		ushort[] arrID4;
        ushort[] arrID1;
        ushort[] arrID3;
		bool[] badPixelArr;
        byte[] imgBuffer;

        ushort[] gMode = new ushort[1000];

        ushort[,] palleteArr = new ushort[1001,3];//-> ushort
		
        ushort gModePeakIx = 0;
        ushort gModePeakCnt = 0;
        ushort gModeLeft = 0;
        ushort gModeRight = 0;
        ushort gModeLeftManual = 0;
        ushort gModeRightManual = 0;
        ushort avgID4 = 0;
        ushort maxTempRaw = 0;

        double gfMaxTempVal = fMIN_TEMP_LIMIT;
        double gfMinTempVal = fMAX_TEMP_LIMIT;
        double[] gfPntTempVal = { 0.0, 0.0, 0.0 };

        System.Drawing.Point gMaxPnt = new System.Drawing.Point(-1, -1);
        System.Drawing.Point gMinPnt = new System.Drawing.Point(-1, -1);
        System.Drawing.Point[] gPoint = { new System.Drawing.Point(-1, -1), 
                                            new System.Drawing.Point(-1, -1), 
                                            new System.Drawing.Point(-1, -1) };
		int		m_DragPntIdx = -1;

        int gTurnAngle = 0;

        double[] gainCalArr;

        Bitmap bitmap;
        Bitmap croppedBitmap;
        Crop cropFilter;

        Bitmap bigBitmap = new Bitmap(IMG_WND_WIDTH, IMG_WND_HEIGHT, PixelFormat.Format24bppRgb);
        BitmapData bitmap_data;
		
        ResizeBilinear bilinearResize = new ResizeBilinear(IMG_WND_WIDTH, IMG_WND_HEIGHT);
        Sharpen sfilter = new Sharpen();

        CIniFile m_IniFile;
        int     nPaletteIndex;

		Bitmap	m_TrendBmp;

        VideoWriter     m_AviFile;
		bool			m_bAviFileOpen = false;

		StreamWriter	m_CsvFile;
		bool			m_bCsvFileOpen = false;

		int				m_RecFrameCount;
		int				m_FrameCount;

        double[,]	m_PntValArr = new double[POINT_NUM, PNT_VAL_NUM];
		int			m_PntValNum = 0;
		int			m_PntValIdx = 0;

		double	m_RangeMax = fMIN_TEMP_LIMIT;
		double	m_RangeMin = fMAX_TEMP_LIMIT;

		int		m_TrendMax = MIN_TREND_LIMIT;
		int		m_TrendMin = MAX_TREND_LIMIT;
		int		m_TrendSize = 0;

		int		m_MaxY = TREND_Y_MIN;
		int		m_MinY = TREND_Y_MAX;

		int		m_TrendHeight;
		int		m_TrendStrH;
		int		m_TrendNameX;
		int		m_TrendValX;
		int		m_TrendPosX;

		bool[]	m_bAlarmStat = new bool[CGlobal.PNT_IDX_NUM] { false, false, false, false, false };

		bool	m_bAlarmEnable = true;
		bool	m_bAlarmBeep = false;
		bool	m_bPlayBeep = false;

		String[] gValName = { "Max", "Min", "Point1", "Point2", "Point3" };

		Pen				m_gridPen = new Pen(Color.Gray);
		Pen[]			m_valPen = { new Pen(Color.Red), 
										new Pen(Color.Blue), 
										new Pen(Color.Orange), 
										new Pen(Color.Green), 
										new Pen(Color.Cyan) };
		
		SolidBrush[]	m_valBrush = { new SolidBrush(Color.Red), 
										new SolidBrush(Color.Blue), 
										new SolidBrush(Color.Orange), 
										new SolidBrush(Color.Green), 
										new SolidBrush(Color.Cyan)};

		SolidBrush		m_AlarmBrush = new SolidBrush(Color.Red);
		SolidBrush		m_AlarmValBrush = new SolidBrush(Color.White);
        Font			m_TrendFnt = new Font("MS Gothic", 9);
        Font			m_PntFnt = new Font("Arial", 9);

        Pen				m_PointPen = new Pen(Color.White);
        Pen				m_ShadowPen = new Pen(Color.Black);

		CGlobal.stSettings m_Settings;

        class ComboItem
        {
            public string Key { get; set; }
            public string Value { get; set; }
            public ComboItem(string key, string value)
            {
                Key = key; Value = value;
            }
            public override string ToString()
            {
                return Value;
            }
        }

        //メッセージ・ハンドラ
        [System.Security.Permissions.PermissionSet(System.Security.Permissions.SecurityAction.Demand, Name = "FullTrust")]
        protected override void WndProc(ref Message m)
        {
            if (m.Msg == WM_THREAD_MSG)
            {
                switch ((uint)m.WParam)
                {
                    case WP_THREAD_EXT_CAL:
                        MenuItemEXTCal.Checked = true;
                        MenuItemINTCal.Checked = false;
                        break;
                }
            }
            base.WndProc(ref m);
        }

        private void InitData(string sModuleDir)
        {
			int	i;

            m_hWnd = IntPtr.Zero;

			//gCsrPntNum = 0;
			m_Settings.TempUnit = "C";
			m_Settings.TrendRangeAuto = true;
			m_Settings.TrendRangeMax = 0.0;
			m_Settings.TrendRangeMin = 0.0;

			m_Settings.AlarmTemp = new CGlobal.stAlarm[CGlobal.PNT_IDX_NUM];

			for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
				m_Settings.AlarmTemp[i].AlarmEnable = false;
				m_Settings.AlarmTemp[i].AlarmTemp = 0.0;
				m_Settings.AlarmTemp[i].AlarmUpLow = CGlobal.ALARM_TYPE_OVER;
				m_Settings.AlarmTemp[i].AlarmHold = CGlobal.ALARM_STATUS_CLEAR;
				m_Settings.AlarmTemp[i].AlarmBeep = false;
			}
            m_Settings.DataSaveDir = sModuleDir + @"\export";
			m_Settings.TempFileName = "SeekCamMon." + CGlobal.TEMP_FILE_EXT;
			m_Settings.VideoFileName = "SeekCamMon." + CGlobal.VIDEO_FILE_EXT;
			m_Settings.SaveTempFile = false;
			m_Settings.SaveVideoFile = false;
			m_Settings.SaveRecCount = 0;
        }

        public Form1()
        {
            InitializeComponent();

            string path;
            string strSfx = "";

            //自分自身のバージョン情報を取得する
            System.Diagnostics.FileVersionInfo ver;
            ver = System.Diagnostics.FileVersionInfo.GetVersionInfo(System.Reflection.Assembly.GetExecutingAssembly().Location);
            if (ver.ProductBuildPart > 0)
            {
                int intSfx = Convert.ToInt32('A') + ver.ProductBuildPart - 1;
                strSfx += Convert.ToChar(intSfx);
            }
            sProgVer = ver.ProductMajorPart.ToString("0") + "." + ver.ProductMinorPart.ToString("00") + strSfx;

            sAppName = Application.ProductName;

            //実行ファイルのパスを得る
            path = System.Windows.Forms.Application.ExecutablePath;
            m_sModuleDir = System.IO.Path.GetDirectoryName(path);

            sIniFileName = System.IO.Path.ChangeExtension(path, null) + ".ini";
            m_IniFile = new CIniFile(sIniFileName);

            InitData(m_sModuleDir);

            var device = SeekThermal.Enumerate().FirstOrDefault();
            if(device == null)
            {
                MessageBox.Show("No Seek Thermal devices found.");
                return;
            }
            thermal = new SeekThermal(device);

			if (device.ProductID == 0x0011) {
				//Seek Thermal Compact Pro
				m_ImgWidth = PRO_IMG_WIDTH;
				m_ImgHeight = PRO_IMG_HEIGHT;
				m_ImgCropW = PRO_IMG_CROPPED_WIDTH;
				m_ImgCropH = PRO_IMG_CROPPED_HEIGHT;
				m_MinPosX = PRO_MIN_POS_X;
				m_MinPosY = PRO_MIN_POS_Y;
				m_MaxPosX = PRO_MAX_POS_X;
				m_MaxPosY = PRO_MAX_POS_Y;
			} else {
				//Seek Thermal Compact
				m_ImgWidth = IMG_WIDTH;
				m_ImgHeight = IMG_HEIGHT;
				m_ImgCropW = IMG_CROPPED_WIDTH;
				m_ImgCropH = IMG_CROPPED_HEIGHT;
				m_MinPosX = MIN_POS_X;
				m_MinPosY = MIN_POS_Y;
				m_MaxPosX = MAX_POS_X;
				m_MaxPosY = MAX_POS_Y;
			}
			m_ImgPixSize = m_ImgWidth * m_ImgHeight;
			m_fImgScaleW = (double)IMG_WND_WIDTH / (double)m_ImgCropW;
			m_fImgScaleH = (double)IMG_WND_HEIGHT / (double)m_ImgCropH;

			arrID4 = new ushort[m_ImgPixSize];
			arrID1 = new ushort[m_ImgPixSize];
			arrID3 = new ushort[m_ImgPixSize];
			badPixelArr = new bool[m_ImgPixSize];
			imgBuffer = new byte[m_ImgPixSize * 3];

			gainCalArr = new double[m_ImgPixSize];
			bitmap = new Bitmap(m_ImgWidth, m_ImgHeight, PixelFormat.Format24bppRgb);
			croppedBitmap = new Bitmap(m_ImgCropW, m_ImgCropH, PixelFormat.Format24bppRgb);
			cropFilter = new Crop(new Rectangle(0, 0, m_ImgCropW, m_ImgCropH));

            LoadIniFile();

            try
            {
                Directory.CreateDirectory(m_Settings.DataSaveDir);
            }
            catch
            {
                m_Settings.DataSaveDir = m_sModuleDir + @"\export";
                Directory.CreateDirectory(m_Settings.DataSaveDir);
            }
			m_pauseEvent = new ManualResetEvent(true);

            thermalThread = new Thread(ThermalThreadProc);
            thermalThread.IsBackground = true;
            thermalThread.Start();
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            var pngFiles = new DirectoryInfo(Directory.GetCurrentDirectory() + "\\palette").GetFiles("*.png");

            foreach (FileInfo file in pngFiles)
            {
                CombpPal.Items.Add(new ComboItem(file.FullName, file.Name.Replace(".png", "")));
            }
            this.Text = sAppName + " Ver." + sProgVer;

            if (0 <= nPaletteIndex && nPaletteIndex < CombpPal.Items.Count)
            {
                CombpPal.SelectedIndex = nPaletteIndex;
            }
            else
            {
                CombpPal.SelectedIndex = 1;
                nPaletteIndex = CombpPal.SelectedIndex;
            }
            m_hWnd = this.Handle;

            MenuItemMinTemp.Checked = showMinVal;
            MenuItemMaxTemp.Checked = showMaxVal;

            MenuItemSharpnessFilter.Checked = SharpenImage;
            MenuItemAutoRange.Checked = autoRange;
            if (autoRange)
            {
                dynSliders = false;
                MenuItemRelativeSlider.Visible = false;
            }
            else
            {
                MenuItemRelativeSlider.Visible = true;
            }
            MenuItemRelativeSlider.Checked = dynSliders;

            switch (m_Settings.TempUnit) {
                case "C":
                    MenuItemUnitK.Checked = false;
                    MenuItemUnitF.Checked = false;
                    MenuItemUnitC.Checked = true;
					lbl_RangeMax.Text = String.Format("{0,6:0.0}", TEMP_RANGE_MAX);
					lbl_RangeMin.Text = String.Format("{0,6:0.0}", TEMP_RANGE_MIN);
                    break;
                case "F":
                    MenuItemUnitK.Checked = false;
                    MenuItemUnitC.Checked = false;
                    MenuItemUnitF.Checked = true;
					lbl_RangeMax.Text = String.Format("{0,6:0.0}", CtoF(TEMP_RANGE_MAX));
					lbl_RangeMin.Text = String.Format("{0,6:0.0}", CtoF(TEMP_RANGE_MIN));
                    break;
                default:
                    MenuItemUnitC.Checked = false;
                    MenuItemUnitF.Checked = false;
                    MenuItemUnitK.Checked = true;
					lbl_RangeMax.Text = String.Format("{0,6:0.0}", CtoK(TEMP_RANGE_MAX));
					lbl_RangeMin.Text = String.Format("{0,6:0.0}", CtoK(TEMP_RANGE_MIN));
                    break;
            }
            lblUnit.Text = "°" + m_Settings.TempUnit;

            MenuItemEXTCal.Checked = false;
            MenuItemINTCal.Checked = true;

	        m_TrendBmp = new Bitmap(pictTrend.Width, pictTrend.Height, PixelFormat.Format24bppRgb);

			m_PointPen.DashStyle = System.Drawing.Drawing2D.DashStyle.Solid;
			m_ShadowPen.DashStyle = System.Drawing.Drawing2D.DashStyle.Solid;
			m_gridPen.DashStyle = DashStyle.Dot;

            Graphics graph = Graphics.FromImage(m_TrendBmp);

			String	str;
            SizeF	size;
			int		i;
			int		x, name_w;
			int		val_w;

			//str = String.Format("{0:0000.0}", 0.0);
			str = String.Format(" {0,6:F1} ", 0.0);
			size = graph.MeasureString(str, m_TrendFnt);
			val_w = (int)size.Width;

			m_TrendStrH = (int)size.Height;
			m_TrendNameX = val_w;
			m_TrendHeight = m_TrendBmp.Height - (int)(size.Height + 2.0);

			name_w = 0;
			for (i = 0; i < 3; i++) {
				size = graph.MeasureString(gValName[i], m_TrendFnt);
				x = (int)(size.Width);
				if (x > name_w) {
					name_w = x;
				}
			}
			m_TrendValX = m_TrendNameX + name_w;
			m_TrendPosX = m_TrendValX + val_w + 1;

            chk_CsvFile.Text = CGlobal.TEMP_FILE_TYPE;
            chk_AviFile.Text = CGlobal.VIDEO_FILE_TYPE;

			chk_CsvFile.Checked = m_Settings.SaveTempFile;
			chk_AviFile.Checked = m_Settings.SaveVideoFile;
			num_RecFrameCount.Value = m_Settings.SaveRecCount;

            lbl_FrameCount.Text = "";

			BtnStopAlarm.Text = "Stop Ararm";

			timer1.Interval = 100;
			timer1.Start();
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
			m_Settings.SaveTempFile = chk_CsvFile.Checked;
			m_Settings.SaveVideoFile = chk_AviFile.Checked;
			m_Settings.SaveRecCount = Decimal.ToInt32(num_RecFrameCount.Value);

            SaveIniFile();

            stopThread = true;
            if (thermal != null)
            {
				m_pauseEvent.Set();

                thermalThread.Join(500);
                thermal.Deinit();
            }
			m_TrendBmp.Dispose();
			m_gridPen.Dispose();
			m_valPen[0].Dispose();
			m_valPen[1].Dispose();
			m_valPen[2].Dispose();
			m_valBrush[0].Dispose();
			m_valBrush[1].Dispose();
			m_valBrush[2].Dispose();
			m_TrendFnt.Dispose();
			m_PntFnt.Dispose();
            m_PointPen.Dispose();
            m_ShadowPen.Dispose();
        }

        private void BtnStartStop_Click(object sender, EventArgs e)
        {
            if (m_isRunning)
            {
				m_pauseEvent.Reset();
                BtnStartStop.Text = "Start Capture";
            }
            else
            {
				m_TrendMax = MIN_TREND_LIMIT;
				m_TrendMin = MAX_TREND_LIMIT;
				m_RangeMax = fMIN_TEMP_LIMIT;
				m_RangeMin = fMAX_TEMP_LIMIT;

				m_PntValNum = 0;
				m_PntValIdx = 0;

				m_MaxY = TREND_Y_MIN;
				m_MinY = TREND_Y_MAX;

				m_pauseEvent.Set();
                BtnStartStop.Text = "Stop Capture";
            }
            m_isRunning = !m_isRunning;
        }

        private void BtnSaveImage_Click(object sender, EventArgs e)
        {
            saveImage = true;
        }

        private void TrackRangeMax_Scroll(object sender, EventArgs e)
        {
            if (!autoRange)
            {
                if (TrackRangeMax.Value > TrackRangeMin.Value + 10)
                {
                    gModeRightManual = (ushort)TrackRangeMax.Value;
                }
                else
                {
                    TrackRangeMax.Value = TrackRangeMin.Value + 10;
                }
            }
        }

        private void TrackRangeMin_Scroll(object sender, EventArgs e)
        {
            if (!autoRange)
            {
                if (TrackRangeMin.Value < TrackRangeMax.Value - 10)
                {
                    gModeLeftManual = (ushort)TrackRangeMin.Value;
                }
                else
                {
                    TrackRangeMin.Value = TrackRangeMax.Value - 10;
                }
            }
        }

        private void DynSlidres_Changed()
        {
            //dynSliders = !dynSliders;

            int currentLeftPos = TrackRangeMin.Value;
            int currentRightPos = TrackRangeMax.Value;
            int currentDiff = currentRightPos - currentLeftPos;

            if (dynSliders)
            {
                MenuItemRelativeSlider.Checked = true;

                if (currentLeftPos - currentDiff > 4000) TrackRangeMin.Minimum = currentLeftPos - currentDiff;//left min
                else TrackRangeMin.Minimum = 4000;

                if (currentLeftPos + currentDiff * 2 < 20000) TrackRangeMin.Maximum = currentLeftPos + currentDiff * 2;//left max
                else TrackRangeMin.Maximum = 20000;

                if (currentRightPos - currentDiff * 2 > 4000) TrackRangeMax.Minimum = currentRightPos - currentDiff * 2;//right min
                else TrackRangeMax.Minimum = 4000;

                if (currentRightPos + currentDiff < 20000) TrackRangeMax.Maximum = currentRightPos + currentDiff;//right max
                else TrackRangeMax.Maximum = 20000;
            }
            else
            {
                MenuItemRelativeSlider.Checked = false;

                TrackRangeMax.Minimum = 4000;
                TrackRangeMin.Minimum = 4000;
                TrackRangeMax.Maximum = 20000;
                TrackRangeMin.Maximum = 20000;
            }
        }

        private void ComboPal_SelectedIndexChanged(object sender, EventArgs e)
        {
            ComboItem newPal = (ComboItem)CombpPal.SelectedItem;
            Bitmap paletteImg = new Bitmap(newPal.Key);
            Color picColor;

            for (int i = 0; i < 1001; i++) {
                picColor = paletteImg.GetPixel(i, 0);
                palleteArr[i, 0] = picColor.R;
                palleteArr[i, 1] = picColor.G;
                palleteArr[i, 2] = picColor.B;
            }

            paletteImg.RotateFlip(RotateFlipType.Rotate270FlipNone);
            pictColorBar.Image = paletteImg;

            nPaletteIndex = CombpPal.SelectedIndex;
        }

        private void PictImage_MouseClick(object sender, MouseEventArgs e)
        {
			int	i, j;
			int	x, y;

			x = (int)((double)e.X / m_fImgScaleW);
			y = (int)((double)e.Y / m_fImgScaleH);

            if (e.Button == MouseButtons.Left) {
				for (i = 0; i < 3; i++) {
					if (gPoint[i].X >= 0 && gPoint[i].Y >= 0) {
						if (gPoint[i].X - 2 <= x && x <= gPoint[i].X + 2 &&
								gPoint[i].Y - 2 <= y && y <= gPoint[i].Y + 2) {
							break;
						}
					} else if (gPoint[i].X < 0 && gPoint[i].Y < 0) {
						gPoint[i].X = x;
						gPoint[i].Y = y;
						if (gTurnAngle > 0) {
							gPoint[i] = RotatePoint(gPoint[i], bitmap.Width, bitmap.Height, gTurnAngle);
						}
						break;
					}
				}
            } else if (e.Button == MouseButtons.Right) {
				System.Drawing.Point pnt;

				for (i = 0; i < 3; i++) {
					if (gPoint[i].X >= 0 && gPoint[i].Y >= 0) {
						if (gTurnAngle > 0) {
							pnt = RotatePoint(gPoint[i], bitmap.Width, bitmap.Height, gTurnAngle);
						} else {
							pnt = gPoint[i];
						}
						if (pnt.X - 2 <= x && x <= pnt.X + 2 && pnt.Y - 2 <= y && y <= pnt.Y + 2) {
							gPoint[i].X = -1;
							gPoint[i].Y = -1;
							for (j = 0; j < PNT_VAL_NUM; j++) {
								m_PntValArr[CGlobal.PNT_PNT1_IDX + i, j] = fMIN_TEMP_LIMIT;
							}
							break;
						}
					}
				}
            }
        }

		private void PictImage_MouseDown(object sender, MouseEventArgs e) {
			int	i;
			int	x, y;

			x = (int)((double)e.X / m_fImgScaleW);
			y = (int)((double)e.Y / m_fImgScaleH);

            if (e.Button == MouseButtons.Left) {
				for (i = 0; i < 3; i++) {
					if (gPoint[i].X >= 0 && gPoint[i].Y >= 0) {
						System.Drawing.Point pnt = gPoint[i];

						if (gTurnAngle > 0) {
							pnt = RotatePoint(gPoint[i], bitmap.Width, bitmap.Height, gTurnAngle);
						} else {
							pnt = gPoint[i];
						}
						if (pnt.X - 2 <= x && x <= pnt.X + 2 &&
								pnt.Y - 2 <= y && y <= pnt.Y + 2) {
							m_DragPntIdx = i;
							break;
						}
					}
				}
            }
		}

		private void PictImage_MouseUp(object sender, MouseEventArgs e) {
			if (e.Button == MouseButtons.Left) {
				if (m_DragPntIdx >= 0) {
					int	x, y;

					x = (int)((double)e.X / m_fImgScaleW);
					y = (int)((double)e.Y / m_fImgScaleH);

					System.Drawing.Point pnt = new System.Drawing.Point(x, y);

					if (gTurnAngle > 0) {
						pnt = RotatePoint(pnt, bitmap.Width, bitmap.Height, gTurnAngle);
					}
					gPoint[m_DragPntIdx] = pnt;
					m_DragPntIdx = -1;

					PictImage.Invalidate();
				}
			}
		}

		private void PictImage_MouseMove(object sender, MouseEventArgs e) {
            if (e.Button == MouseButtons.Left) {
				if (m_DragPntIdx >= 0) {
					int	x, y;

					x = (int)((double)e.X / m_fImgScaleW);
					y = (int)((double)e.Y / m_fImgScaleH);

					System.Drawing.Point pnt = new System.Drawing.Point(x, y);

					if (gTurnAngle > 0) {
						pnt = RotatePoint(pnt, bitmap.Width, bitmap.Height, gTurnAngle);
					}
					gPoint[m_DragPntIdx] = pnt;

					PictImage.Invalidate();
				}
            }
		}

        private void MenuItemExitApp_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void MenuItemMinTemp_Click(object sender, EventArgs e)
        {
            showMinVal = !showMinVal;

            if (showMinVal)
            {
                MenuItemMinTemp.Checked = true;
            }
            else
            {
                MenuItemMinTemp.Checked = false;
            }
        }

        private void MenuItemMaxTemp_Click(object sender, EventArgs e)
        {
            showMaxVal = !showMaxVal;

            if (showMaxVal)
            {
                MenuItemMaxTemp.Checked = true;
            }
            else
            {
                MenuItemMaxTemp.Checked = false;
            }
        }

        private void MenuItemAutoRange_Click(object sender, EventArgs e)
        {
            autoRange = !autoRange;
            if (autoRange)
            {
                MenuItemAutoRange.Checked = true;
                MenuItemRelativeSlider.Visible = false;
				dynSliders = false;
				DynSlidres_Changed();

				m_RangeMax = fMIN_TEMP_LIMIT;
				m_RangeMin = fMAX_TEMP_LIMIT;
            }
            else
            {
                MenuItemAutoRange.Checked = false;
                //MenuItemRelativeSlider.Checked = false;
                MenuItemRelativeSlider.Visible = true;
				DynSlidres_Changed();
            }
       }

        private void MenuItemSharpnessFilter_Click(object sender, EventArgs e)
        {
            SharpenImage = !SharpenImage;
            if (SharpenImage)
            {
                MenuItemSharpnessFilter.Checked = true;
            }
            else
            {
                MenuItemSharpnessFilter.Checked = false;
            }
        }

        private void MenuItemRelativeSlider_Click(object sender, EventArgs e)
        {
            dynSliders = !dynSliders;

			DynSlidres_Changed();
        }

        private void MenuItemUnitK_Click(object sender, EventArgs e)
        {
			int	i;

			if (m_Settings.TempUnit != "K") {
				switch (m_Settings.TempUnit) {
				case "F":
					if (!m_Settings.TrendRangeAuto) {
						m_Settings.TrendRangeMax = FtoK(m_Settings.TrendRangeMax);
						m_Settings.TrendRangeMin = FtoK(m_Settings.TrendRangeMin);
					}
					for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
						if (m_Settings.AlarmTemp[i].AlarmEnable) {
							m_Settings.AlarmTemp[i].AlarmTemp = FtoK(m_Settings.AlarmTemp[i].AlarmTemp);
						}
					}
					break;
				case "C":
					if (!m_Settings.TrendRangeAuto) {
						m_Settings.TrendRangeMax = CtoK(m_Settings.TrendRangeMax);
						m_Settings.TrendRangeMin = CtoK(m_Settings.TrendRangeMin);
					}
					for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
						if (m_Settings.AlarmTemp[i].AlarmEnable) {
							m_Settings.AlarmTemp[i].AlarmTemp = CtoK(m_Settings.AlarmTemp[i].AlarmTemp);
						}
					}
					break;
				}
				m_Settings.TempUnit = "K";
				lbl_RangeMax.Text = String.Format("{0,6:0.0}", CtoK(TEMP_RANGE_MAX));
				lbl_RangeMin.Text = String.Format("{0,6:0.0}", CtoK(TEMP_RANGE_MIN));

				MenuItemUnitC.Checked = false;
				MenuItemUnitF.Checked = false;
				MenuItemUnitK.Checked = true;
				lblUnit.Text = "°" + m_Settings.TempUnit;

				m_RangeMax = fMIN_TEMP_LIMIT;
				m_RangeMin = fMAX_TEMP_LIMIT;
				m_PntValNum = 0;
				m_PntValIdx = 0;
			}
        }

        private void MenuItemUnitC_Click(object sender, EventArgs e)
        {
			int	i;

			if (m_Settings.TempUnit != "C") {
				switch (m_Settings.TempUnit) {
				case "K":
					if (!m_Settings.TrendRangeAuto) {
						m_Settings.TrendRangeMax = KtoC(m_Settings.TrendRangeMax);
						m_Settings.TrendRangeMin = KtoC(m_Settings.TrendRangeMin);
					}
					for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
						if (m_Settings.AlarmTemp[i].AlarmEnable) {
							m_Settings.AlarmTemp[i].AlarmTemp = KtoC(m_Settings.AlarmTemp[i].AlarmTemp);
						}
					}
					break;
				case "F":
					if (!m_Settings.TrendRangeAuto) {
						m_Settings.TrendRangeMax = FtoC(m_Settings.TrendRangeMax);
						m_Settings.TrendRangeMin = FtoC(m_Settings.TrendRangeMin);
					}
					for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
						if (m_Settings.AlarmTemp[i].AlarmEnable) {
							m_Settings.AlarmTemp[i].AlarmTemp = FtoC(m_Settings.AlarmTemp[i].AlarmTemp);
						}
					}
					break;
				}
				m_Settings.TempUnit = "C";
				lbl_RangeMax.Text = String.Format("{0,6:0.0}", TEMP_RANGE_MAX);
				lbl_RangeMin.Text = String.Format("{0,6:0.0}", TEMP_RANGE_MIN);

				MenuItemUnitK.Checked = false;
				MenuItemUnitF.Checked = false;
				MenuItemUnitC.Checked = true;
				lblUnit.Text = "°" + m_Settings.TempUnit;

				m_RangeMax = fMIN_TEMP_LIMIT;
				m_RangeMin = fMAX_TEMP_LIMIT;
				m_PntValNum = 0;
				m_PntValIdx = 0;
			}
        }

        private void MenuItemUnitF_Click(object sender, EventArgs e)
        {
			int	i;

			if (m_Settings.TempUnit != "F") {
				switch (m_Settings.TempUnit) {
				case "K":
					if (!m_Settings.TrendRangeAuto) {
						m_Settings.TrendRangeMax = KtoF(m_Settings.TrendRangeMax);
						m_Settings.TrendRangeMin = KtoF(m_Settings.TrendRangeMin);
					}
					for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
						if (m_Settings.AlarmTemp[i].AlarmEnable) {
							m_Settings.AlarmTemp[i].AlarmTemp = KtoF(m_Settings.AlarmTemp[i].AlarmTemp);
						}
					}
					break;
				case "C":
					if (!m_Settings.TrendRangeAuto) {
						m_Settings.TrendRangeMax = CtoF(m_Settings.TrendRangeMax);
						m_Settings.TrendRangeMin = CtoF(m_Settings.TrendRangeMin);
					}
					for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
						if (m_Settings.AlarmTemp[i].AlarmEnable) {
							m_Settings.AlarmTemp[i].AlarmTemp = CtoF(m_Settings.AlarmTemp[i].AlarmTemp);
						}
					}
					break;
				}
				m_Settings.TempUnit = "F";
				lbl_RangeMax.Text = String.Format("{0,6:0.0}", CtoF(TEMP_RANGE_MAX));
				lbl_RangeMin.Text = String.Format("{0,6:0.0}", CtoF(TEMP_RANGE_MIN));

				MenuItemUnitK.Checked = false;
				MenuItemUnitC.Checked = false;
				MenuItemUnitF.Checked = true;
				lblUnit.Text = "°" + m_Settings.TempUnit;

				m_RangeMax = fMIN_TEMP_LIMIT;
				m_RangeMin = fMAX_TEMP_LIMIT;
				m_PntValNum = 0;
				m_PntValIdx = 0;
			}
        }

        private void MenuItemINTCal_Click(object sender, EventArgs e)
        {
            usignExternalCal = false;
            MenuItemEXTCal.Checked = false;
            MenuItemINTCal.Checked = true;
        }

        private void MenuItemEXTCal_Click(object sender, EventArgs e)
        {
            grabExternalReference = true;
        }

        private void MenuItemTurn180_Click(object sender, EventArgs e)
        {
            if (gTurnAngle == 180)
            {
                gTurnAngle = 0;
                MenuItemTurn180.Checked = false;
            }
            else
            {
                gTurnAngle = 180;
                MenuItemTurn180.Checked = true;
            }
        }

        private void LoadIniFile()
        {
            string  strVal;
            bool    bVal;
            int     iVal;
			double	fVal;
			int		i;
			string	strKey;

            strVal = m_IniFile["SETTING", "SHOW_MIN"];
            if (strVal == "") {
                m_IniFile["SETTING", "SHOW_MIN"] = showMinVal.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    showMinVal = bVal;
                }
            }

            strVal = m_IniFile["SETTING", "SHOW_MAX"];
            if (strVal == "") {
                m_IniFile["SETTING", "SHOW_MAX"] = showMaxVal.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    showMaxVal = bVal;
                }
            }

            strVal = m_IniFile["SETTING", "SHARPEN_IMAGE"];
            if (strVal == "") {
                m_IniFile["SETTING", "SHARPEN_IMAGE"] = SharpenImage.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    SharpenImage = bVal;
                }
            }

            strVal = m_IniFile["SETTING", "AUTO_RANGE"];
            if (strVal == "") {
                m_IniFile["SETTING", "AUTO_RANGE"] = autoRange.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    autoRange = bVal;
                }
            }

            strVal = m_IniFile["SETTING", "TEMP_UNIT"];
            if (strVal == "") {
                m_IniFile["SETTING", "TEMP_UNIT"] = m_Settings.TempUnit;
            } else {
                if (strVal == "K" || strVal == "C" || strVal == "F") {
                    m_Settings.TempUnit = strVal;
                }
            }

            strVal = m_IniFile["SETTING", "PALETTE"];
            if (strVal == "") {
                m_IniFile["SETTING", "PALETTE"] = nPaletteIndex.ToString();
            } else {
                if (int.TryParse(strVal, out iVal)) {
                    nPaletteIndex = iVal;
                }
            }

			for (i = 0; i < 3; i++) {
				strKey = "POINT_" + (i + 1).ToString() + "_X";
				strVal = m_IniFile["SETTING", strKey];
				if (strVal == "") {
					m_IniFile["SETTING", strKey] = gPoint[i].X.ToString();
				} else {
					if (int.TryParse(strVal, out iVal)) {
						if (m_MinPosX < iVal && iVal < m_MaxPosX) {
							gPoint[i].X = iVal;
						}
					}
				}
				strKey = "POINT_" + (i + 1).ToString() + "_Y";
				strVal = m_IniFile["SETTING", strKey];
				if (strVal == "") {
					m_IniFile["SETTING", strKey] = gPoint[i].Y.ToString();
				} else {
					if (int.TryParse(strVal, out iVal)) {
						if (m_MinPosY < iVal && iVal < m_MaxPosY) {
							gPoint[i].Y = iVal;
						}
					}
				}
			}

            strVal = m_IniFile["SETTING", "TREND_RANGE_AUTO"];
            if (strVal == "") {
                m_IniFile["SETTING", "TREND_RANGE_AUTO"] = m_Settings.TrendRangeAuto.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    m_Settings.TrendRangeAuto = bVal;
                }
            }
            strVal = m_IniFile["SETTING", "TREND_RANGE_MAX"];
            if (strVal == "") {
                m_IniFile["SETTING", "TREND_RANGE_MAX"] = String.Format("{0:0.0}", m_Settings.TrendRangeMax);
            } else {
                if (double.TryParse(strVal, out fVal)) {
                    m_Settings.TrendRangeMax = fVal;
                }
            }
            strVal = m_IniFile["SETTING", "TREND_RANGE_MIN"];
            if (strVal == "") {
                m_IniFile["SETTING", "TREND_RANGE_MIN"] = String.Format("{0:0.0}", m_Settings.TrendRangeMin);
            } else {
                if (double.TryParse(strVal, out fVal)) {
                    m_Settings.TrendRangeMin = fVal;
                }
            }
			for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
				strKey = "AlarmEnable" + gValName[i];
	            strVal = m_IniFile["SETTING", strKey];
				if (strVal == "") {
					m_IniFile["SETTING", strKey] = m_Settings.AlarmTemp[i].AlarmEnable.ToString();
				} else {
					if (bool.TryParse(strVal, out bVal)) {
						m_Settings.AlarmTemp[i].AlarmEnable = bVal;
					}
				}
				strKey = "AlarmTemp" + gValName[i];
	            strVal = m_IniFile["SETTING", strKey];
				if (strVal == "") {
					m_IniFile["SETTING", strKey] = String.Format("{0:0.0}", m_Settings.AlarmTemp[i].AlarmTemp);
				} else {
					if (double.TryParse(strVal, out fVal)) {
						m_Settings.AlarmTemp[i].AlarmTemp = fVal;
					}
				}
				strKey = "AlarmUpLow" + gValName[i];
	            strVal = m_IniFile["SETTING", strKey];
				if (strVal == "") {
					m_IniFile["SETTING", strKey] = m_Settings.AlarmTemp[i].AlarmUpLow.ToString();
				} else {
					if (int.TryParse(strVal, out iVal)) {
						m_Settings.AlarmTemp[i].AlarmUpLow = iVal;
					}
				}
				strKey = "AlarmHold" + gValName[i];
	            strVal = m_IniFile["SETTING", strKey];
				if (strVal == "") {
					m_IniFile["SETTING", strKey] = m_Settings.AlarmTemp[i].AlarmHold.ToString();
				} else {
					if (int.TryParse(strVal, out iVal)) {
						m_Settings.AlarmTemp[i].AlarmHold = iVal;
					}
				}
			}

            strVal = m_IniFile["SETTING", "DATA_SAVE_DIR"];
            if (strVal != "") {
                m_Settings.DataSaveDir = strVal;
            } else {
                m_Settings.DataSaveDir = m_sModuleDir + @"\export";
                m_IniFile["SETTING", "DATA_SAVE_DIR"] = m_Settings.DataSaveDir;
            }

            strVal = m_IniFile["SETTING", "TEMP_FILE_NAME"];
            if (strVal != "") {
                m_Settings.TempFileName = strVal;
            } else {
                m_Settings.TempFileName = @"SeekCamMon." + CGlobal.TEMP_FILE_EXT;
                m_IniFile["SETTING", "TEMP_FILE_NAME"] = m_Settings.TempFileName;
            }
   
			strVal = m_IniFile["SETTING", "VIDEO_FILE_NAME"];
            if (strVal != "") {
                m_Settings.VideoFileName = strVal;
            } else {
                m_Settings.VideoFileName = @"SeekCamMon." + CGlobal.VIDEO_FILE_EXT;
                m_IniFile["SETTING", "VIDEO_FILE_NAME"] = m_Settings.VideoFileName;
            }

            strVal = m_IniFile["SETTING", "SAVE_TEMP_FILE"];
            if (strVal == "") {
                m_IniFile["SETTING", "SAVE_TEMP_FILE"] = m_Settings.SaveTempFile.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    m_Settings.SaveTempFile = bVal;
                }
            }

            strVal = m_IniFile["SETTING", "SAVE_VIDEO_FILE"];
            if (strVal == "") {
                m_IniFile["SETTING", "SAVE_VIDEO_FILE"] = m_Settings.SaveVideoFile.ToString();
            } else {
                if (bool.TryParse(strVal, out bVal)) {
                    m_Settings.SaveVideoFile = bVal;
                }
            }

            strVal = m_IniFile["SETTING", "SAVE_REC_COUNT"];
            if (strVal == "") {
                m_IniFile["SETTING", "SAVE_REC_COUNT"] = m_Settings.SaveRecCount.ToString();
            } else {
                if (int.TryParse(strVal, out iVal)) {
                    m_Settings.SaveRecCount = iVal;
                }
            }
        }

        private void SaveIniFile()
        {
			int		i;
			string	strKey;

            m_IniFile["SETTING", "SHOW_MIN"] = showMinVal.ToString();
            m_IniFile["SETTING", "SHOW_MAX"] = showMaxVal.ToString();
            m_IniFile["SETTING", "SHARPEN_IMAGE"] = SharpenImage.ToString();
            m_IniFile["SETTING", "AUTO_RANGE"] = autoRange.ToString();
            m_IniFile["SETTING", "TEMP_UNIT"] = m_Settings.TempUnit;
            m_IniFile["SETTING", "PALETTE"] = nPaletteIndex.ToString();

			for (i = 0; i < 3; i++) {
				strKey = "POINT_" + (i + 1).ToString() + "_X";
				m_IniFile["SETTING", strKey] = gPoint[i].X.ToString();
				
				strKey = "POINT_" + (i + 1).ToString() + "_Y";
				m_IniFile["SETTING", strKey] = gPoint[i].Y.ToString();
			}
            m_IniFile["SETTING", "TREND_RANGE_AUTO"] = m_Settings.TrendRangeAuto.ToString();
            m_IniFile["SETTING", "TREND_RANGE_MAX"] = String.Format("{0:0.0}", m_Settings.TrendRangeMax);
            m_IniFile["SETTING", "TREND_RANGE_MIN"] = String.Format("{0:0.0}", m_Settings.TrendRangeMin);

			for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
				strKey = "AlarmEnable" + gValName[i];
				m_IniFile["SETTING", strKey] = m_Settings.AlarmTemp[i].AlarmEnable.ToString();

				strKey = "AlarmTemp" + gValName[i];
				m_IniFile["SETTING", strKey] = String.Format("{0:0.0}", m_Settings.AlarmTemp[i].AlarmTemp);

				strKey = "AlarmUpLow" + gValName[i];
				m_IniFile["SETTING", strKey] = m_Settings.AlarmTemp[i].AlarmUpLow.ToString();

				strKey = "AlarmHold" + gValName[i];
				m_IniFile["SETTING", strKey] = m_Settings.AlarmTemp[i].AlarmHold.ToString();
			}
            m_IniFile["SETTING", "DATA_SAVE_DIR"] = m_Settings.DataSaveDir;
            m_IniFile["SETTING", "TEMP_FILE_NAME"] = m_Settings.TempFileName;
            m_IniFile["SETTING", "VIDEO_FILE_NAME"] = m_Settings.VideoFileName;

			m_IniFile["SETTING", "SAVE_TEMP_FILE"] = m_Settings.SaveTempFile.ToString();
			m_IniFile["SETTING", "SAVE_VIDEO_FILE"] = m_Settings.SaveVideoFile.ToString();
			m_IniFile["SETTING", "SAVE_REC_COUNT"] = m_Settings.SaveRecCount.ToString();
        }

        void ThermalThreadProc()
        {
            while (!stopThread && thermal != null)
            {
                bool progress = false;

				m_pauseEvent.WaitOne(Timeout.Infinite);

                currentFrame = thermal.GetFrameBlocking();

                switch (currentFrame.StatusByte)
                {
                    case 4://gain calibration
                        Frame4stuff();
                        break;

                    case 1://shutter calibration
                        MarkBadPixels();
                        if (!usignExternalCal)
                        {
                            Frame1stuff();
                        }
                        //firstAfterCal = true;
                        break;
                    
                    case 3://image frame
                        MarkBadPixels();
                        if (grabExternalReference)//use this image as reference
                        {
                            grabExternalReference = false;
                            usignExternalCal = true;

                            Frame1stuff();

                            if (m_hWnd != IntPtr.Zero)
                            {

                                PostMessage(m_hWnd, WM_THREAD_MSG, WP_THREAD_EXT_CAL, 1);
                            }
                        }
                        else
                        {
                            Frame3stuff();
                            lastUsableFrame = currentFrame;
                            progress = true;
                        }
                        break;

                    default:
                        break;
                }

                if(progress)
                {
                    Invalidate();//redraw form
                }
            }
        }

        private void Frame4stuff()
        {
            arrID4 = currentFrame.RawDataU16;
            avgID4 = GetMode(arrID4);

            for (int i = 0; i < IMG_PIX_SIZE; i++)
            {
                if (arrID4[i] > 2000 && arrID4[i] < 8000) {
                    gainCalArr[i] = avgID4 / (double)arrID4[i];
				}
                else {
					gainCalArr[i] = 1;
					badPixelArr[i] = true;
				}
            }
        }

        private void Frame1stuff()
        {
            arrID1 = currentFrame.RawDataU16;
            //avgID1 = GetMode(arrID1);
        }

        private void Frame3stuff()
        {
            arrID3 = currentFrame.RawDataU16;

            for (int i = 0; i < IMG_PIX_SIZE; i++)
            {
                if (arrID3[i] > 2000)
                {
                    arrID3[i] = (ushort)((arrID3[i] - arrID1[i]) * gainCalArr[i] + 7500);
                }
                else
                {
                    arrID3[i] = 0;
                    badPixelArr[i] = true;
                }
            }

            FixBadPixels();
			RemoveNoise();
            GetHistogram();
            FillImgBuffer();
        }

        private double RawToK(int val)
        {
            double tempVal = 0;

            tempVal = (double)(val - 5950) / 40 + 273.15;//K

            return tempVal;
        }

        private double KtoC(double val)
        {
            double tempVal;

            tempVal = val - 273.15;	//C

            return tempVal;
        }

        private double KtoF(double val)
        {
            double tempVal;

            tempVal = (val - 273.15) * 1.8 + 32.0;	//F

            return tempVal;
        }

        private double CtoK(double val)
        {
            double tempVal;

            tempVal = val + 273.15;	//C->K

            return tempVal;
        }

        private double FtoK(double val)
        {
            double tempVal;

            tempVal = (val - 32.0) / 1.8 + 273.15;	//F->K

            return tempVal;
        }

        private double CtoF(double val)
        {
            double tempVal;

            tempVal = (val * 1.8) + 32.0;		//C->F

            return tempVal;
        }

        private double FtoC(double val)
        {
            double tempVal;

            tempVal = (val - 32.0) / 1.8;		//F->C

            return tempVal;
        }

        private double RawToTempVal(int val)
        {
            double tempVal = 0;

            tempVal = RawToK(val);	//K

            switch (m_Settings.TempUnit)
            {
                case "C":
                    tempVal = KtoC(tempVal);	//C
                    break;
                case "F":
                    tempVal = KtoF(tempVal);	//F
                    break;
            }

            return tempVal;
        }

        private string RawToTemp(int val)
        {
            double tempVal = RawToTempVal(val);

            return tempVal.ToString("F1", CultureInfo.InvariantCulture);
        }

        private double GetMinTempLimit()
        {
            double val;

            val = fMIN_TEMP_LIMIT;

            return val;
        }

        private double GetMaxTempLimit()
        {
            double val;

            val = fMAX_TEMP_LIMIT;

            return val;
        }

        private void FillImgBuffer()
        {
            ushort v = 0;
            ushort loc = 0;
            int x = -1;
            int y = -1;
            double fTemp;
			int	i, j;

            double iScaler;
            iScaler = (double)(gModeRight - gModeLeft) / 1000;

            gMinPnt.X = -1;
            gMinPnt.Y = -1;
            gMaxPnt.X = -1;
            gMaxPnt.Y = -1;
            gfMaxTempVal = fMIN_TEMP_LIMIT;
            gfMinTempVal = fMAX_TEMP_LIMIT;
			gfPntTempVal[0] = fMIN_TEMP_LIMIT;
			gfPntTempVal[1] = fMIN_TEMP_LIMIT;
			gfPntTempVal[2] = fMIN_TEMP_LIMIT;

            for (i = 0; i < m_ImgPixSize; i++) {
                x = i % m_ImgWidth;
                y = i / m_ImgWidth;

                v = arrID3[i];

                fTemp = RawToTempVal(v);

				for (j = 0; j < 3; j++) {
					if (gPoint[j].X == x && gPoint[j].Y == y) {
						gfPntTempVal[j] = fTemp;
					}
				}
                if ((m_MinPosY < y && y < m_MaxPosY) && (m_MinPosX < x && x < m_MaxPosX)) {
                    if (fTemp > GetMinTempLimit() && fTemp < gfMinTempVal) {
                        gfMinTempVal = fTemp;
                        gMinPnt.X = x;
                        gMinPnt.Y = y;

                        //System.Diagnostics.Debug.WriteLine("MinVal=" + fTemp.ToString() + " x=" + x.ToString() + " y=" + y.ToString());
                    }
                    if (fTemp < GetMaxTempLimit() && fTemp > gfMaxTempVal) {
                        gfMaxTempVal = fTemp;
                        gMaxPnt.X = x;
                        gMaxPnt.Y = y;

                        //System.Diagnostics.Debug.WriteLine("MaxVal=" + fTemp.ToString() + " x=" + x.ToString() + " y=" + y.ToString());
                    }
                }

                if (v < gModeLeft) {
                    v = gModeLeft;
                }
                if (v > gModeRight) {
                    v = gModeRight;
                }
                v = (ushort)(v - gModeLeft);
                loc = (ushort)(v / iScaler);

                imgBuffer[i*3] = (byte)palleteArr[loc, 2];
                imgBuffer[i*3+1] = (byte)palleteArr[loc, 1];
                imgBuffer[i*3+2] = (byte)palleteArr[loc, 0];
            }
        }

        private void MarkBadPixels()
        {
            ushort[] RawDataArr = currentFrame.RawDataU16;

            for (int i = 0; i < RawDataArr.Length; i++)
            {
                if (RawDataArr[i] < 2000 || RawDataArr[i] > 22000)
                {
                    badPixelArr[i] = true;
                }
            }
        }
		
		private void FixBadPixels()
       {
           ushort x = 0;
           ushort y = 0;
           ushort i = 0;
           ushort nr = 0;
           ushort val = 0;

           for (y = 0; y < m_ImgHeight; y++)
           {
               for (x = 0; x < m_ImgWidth; x++, i++)
               {
                   if (badPixelArr[i] && x < m_ImgCropW) {

                       val = 0;
                       nr = 0;

                       if (y > 0 && !badPixelArr[i - m_ImgWidth]) //top pixel
                       {
                           val += arrID3[i - m_ImgWidth];
                           ++nr;
                       }

                       if (y < (m_ImgHeight - 1) && !badPixelArr[i + m_ImgWidth]) // bottom pixel
                       {
                           val += arrID3[i + m_ImgWidth];
                           ++nr;
                       }

                       if (x > 0 && !badPixelArr[i - 1]) //Left pixel
                       {
                           val += arrID3[i - 1];
                           ++nr;
                       }

                       if (x < (m_ImgCropW - 1) && !badPixelArr[i + 1]) //Right pixel
                       {
                           val += arrID3[i + 1];
                           ++nr;
                       }

                       if (nr>0)
                       {
                           val /= nr;
                           arrID3[i] = val;
                       }
                   }
               }
           }
       }

       private void RemoveNoise()
       {
           ushort x = 0;
           ushort y = 0;
           ushort i = 0;
           ushort val = 0;
           ushort[] arrColor = new ushort[4];

           for (y = 0; y < m_ImgHeight; y++)
           {
               for (x = 0; x < m_ImgWidth; x++)
               {
                   if (x > 0 && x < m_ImgCropW && y > 0 && y < (m_ImgCropH - 1))
                   {
                       arrColor[0] = arrID3[i - m_ImgWidth];//top
                       arrColor[1] = arrID3[i + m_ImgWidth];//bottom
                       arrColor[2] = arrID3[i - 1];//left
                       arrColor[3] = arrID3[i + 1];//right

                       val = (ushort)((arrColor[0] + arrColor[1] + arrColor[2] + arrColor[3] - Highest(arrColor) - Lowest(arrColor))/2);

                       if (Math.Abs(val - arrID3[i]) > 100 && val != 0)
                       {
                           arrID3[i] = val;
                       }
                   }
                   i++;
               }
           }

       }

       private ushort Highest(ushort[] numbers)
       {
           ushort highest = 0;

           for (ushort i = 0; i < 4; i++)
           {
               if (numbers[i] > highest)
                   highest = numbers[i];
           }

           return highest;
       }

       private ushort Lowest(ushort[] numbers)
       {
           ushort lowest = 30000;

           for (ushort i = 0; i < 4; i++)
           {
               if (numbers[i] < lowest)
                   lowest = numbers[i];
           }

           return lowest;
       }

        public ushort GetMode(ushort[] arr)
        {
            ushort[] arrMode = new ushort[320];
            ushort topPos = 0;
            for (ushort i = 0; i < m_ImgPixSize; i++)
            {
                if ((arr[i] > 1000) && (arr[i] / 100 != 0)) arrMode[(arr[i]) / 100]++;
            }

            topPos = (ushort)Array.IndexOf(arrMode, arrMode.Max());

            return (ushort)(topPos * 100);
        }

        public void GetHistogram()
        {
            maxTempRaw = arrID3.Max();
            ushort[] arrMode = new ushort[2100];
            ushort topPos = 0;
            for (ushort i = 0; i < m_ImgPixSize; i++)
            {
                if ((arrID3[i] > 1000) && (arrID3[i] / 10 != 0) && !badPixelArr[i]) arrMode[(arrID3[i]) / 10]++;
            }

            topPos = (ushort)Array.IndexOf(arrMode, arrMode.Max());

            gMode = arrMode;
            gModePeakCnt = arrMode.Max();
            gModePeakIx = (ushort)(topPos * 10);

            //lower it to 100px;
            for (ushort i = 0; i < 2100; i++)
            {
                gMode[i] = (ushort)((double)arrMode[i] / gModePeakCnt * 100);
            }

            if (autoRange)
            {
                gModeLeft = gModePeakIx;
                gModeRight = gModePeakIx;
                //find left border:
                for (ushort i = 0; i < topPos; i++)
                {
                    if (arrMode[i] > arrMode[topPos] * 0.01)
                    {
                        gModeLeft = (ushort)(i * 10);
                        break;
                    }
                }

                //find right border:
                for (ushort i = 2099; i > topPos; i--)
                {
                    if (arrMode[i] > arrMode[topPos] * 0.01)
                    {
                        gModeRight = (ushort)(i * 10);
                        break;
                    }
                }
                gModeLeftManual = gModeLeft;
                gModeRightManual = gModeRight;
            }
            else {
                gModeLeft = gModeLeftManual;
                gModeRight = gModeRightManual;
            }
        }

#if false
        public void DrawHistogram()
        {
            int imgWidth = (gModeRight - gModeLeft)/10;
            int leftBorder = gModeLeft / 10;
            var hist = new Bitmap(imgWidth, 100, PixelFormat.Format24bppRgb);
            Pen blackPen = new Pen(Color.Black, 1);

            using (var g = Graphics.FromImage(hist))
            {
                g.FillRectangle(new SolidBrush(Color.White), 0, 0, imgWidth, 100);

                for (int i = 0; i < imgWidth; i++)
                {
                    g.DrawLine(blackPen, i, 0, i, gMode[leftBorder + i]);
                }
            }

            hist.RotateFlip(RotateFlipType.Rotate180FlipX);
            hist = new Bitmap(hist, new Size(200, 100));

            pictureBox2.Image = hist;
        }
#endif

        private System.Drawing.Point RotatePoint(System.Drawing.Point pnt, int width, int height, int angle)
        {
            System.Drawing.Point   new_pnt = new System.Drawing.Point();
            double  cx, cy;
            double  ox, oy;
            double  nx, ny;
            double  rad;

            rad = (360.0 - (double)angle) * Math.PI / 180.0;

            cx = (double)width / 2;
            cy = (double)height / 2;

            ox = pnt.X - cx;
            oy = (pnt.Y - cy) * -1; //座標系が上下逆の為符号を反転

            nx = (ox * Math.Cos(rad)) - (oy * Math.Sin(rad));
            ny = (oy * Math.Cos(rad)) + (ox * Math.Sin(rad));

            new_pnt.X = (int)(cx + nx);
            new_pnt.Y = (int)(cy - ny);

            if (angle == 90)
            {

            } else if (angle == 270)
            {

            }

            return new_pnt;
        }

		private void Draw_TrendLine(Graphics graph, int no, int idx) {
			double	val = 0.0;
			double	vy;
			int		val_y = 0;
			int		i, j;
			int		x, y;
			int		old_x = -1;
			int		old_y = -1;
            String	str;
            SizeF	size;
            Rectangle   clipRect = new Rectangle(0, 0, IMG_WND_WIDTH, m_TrendHeight);

            graph.SetClip(clipRect, CombineMode.Replace);

			j = idx;
			for (i = 0; i < m_PntValNum; i++) {
				val = m_PntValArr[no, j];

				x = (int)(IMG_WND_WIDTH - m_PntValNum + i);
				vy = val - m_TrendMin;
				val_y = m_TrendHeight - (int)((double)m_TrendHeight * vy / m_TrendSize + 0.5);

				if (old_x >= 0) {
					graph.DrawLine(m_valPen[no], old_x, old_y, x, val_y);
					old_x = x;
				} else {
					if (fMIN_TEMP_LIMIT < val && val < fMAX_TEMP_LIMIT) {
						old_x = x;
					}
				}
				old_y = val_y;
				j++;
				if (j >= PNT_VAL_NUM) {
					j = 0;
				}
			}
            graph.ResetClip();

			if (m_PntValNum > 0) {
				//str = val.ToString("F1", CultureInfo.InvariantCulture);
				size = graph.MeasureString(gValName[0], m_TrendFnt);

				//y = val_y - (int)(size.Height / 2.0 + 0.5);
				//graph.DrawString(str, m_TrendFnt, m_valBrush[no], IMG_WND_WIDTH + 2, y);
				y = (int)(size.Height + 2) * (no + 1);

				x = IMG_WND_WIDTH + m_TrendNameX;
				graph.DrawString(gValName[no], m_TrendFnt, m_valBrush[no], x, y);


				x = IMG_WND_WIDTH + m_TrendValX;
				//str = String.Format(": {0:###0.0}", val);
				str = String.Format(" {0,6:F1} ", val);
				size = graph.MeasureString(str, m_TrendFnt);
				if (m_bAlarmStat[no]) {
					graph.FillRectangle(m_AlarmBrush, x, y, size.Width, size.Height);
					graph.DrawString(str, m_TrendFnt, m_AlarmValBrush, x, y);
				} else {
					graph.DrawString(str, m_TrendFnt, m_valBrush[no], x, y);
				}
				switch (no) {
					case CGlobal.PNT_MAX_IDX:
						str = String.Format("({0,3:0},{1,3:0})", gMaxPnt.X, gMaxPnt.Y);
						break;
					case CGlobal.PNT_PNT1_IDX:
						str = String.Format("({0,3:0},{1,3:0})", gPoint[no - CGlobal.PNT_PNT1_IDX].X, gPoint[no - CGlobal.PNT_PNT1_IDX].Y);
						break;
					case CGlobal.PNT_PNT2_IDX:
						str = String.Format("({0,3:0},{1,3:0})", gPoint[no - CGlobal.PNT_PNT1_IDX].X, gPoint[no - CGlobal.PNT_PNT1_IDX].Y);
						break;
					case CGlobal.PNT_PNT3_IDX:
						str = String.Format("({0,3:0},{1,3:0})", gPoint[no - CGlobal.PNT_PNT1_IDX].X, gPoint[no - CGlobal.PNT_PNT1_IDX].Y);
						break;
					case CGlobal.PNT_MIN_IDX:
						str = String.Format("({0,3:0},{1,3:0})", gMinPnt.X, gMinPnt.Y);
						break;
				}
				x = IMG_WND_WIDTH + m_TrendPosX;
				graph.DrawString(str, m_TrendFnt, m_valBrush[no], x, y);
			}
		}

		private void Draw_Trend()
		{
			double	rngMin = RawToTempVal(gModeLeft);
			double	rngMax = RawToTempVal(gModeRight);
			double	rangeSize = m_RangeMax - m_RangeMin;
			double	trendSize = m_TrendMax - m_TrendMin;

			Graphics	graph = Graphics.FromImage(m_TrendBmp);
			int			idx;
			String		str;
			SizeF		size;
			int			i, x;

			if (m_RangeMax <= fMIN_TEMP_LIMIT || m_RangeMin >= fMAX_TEMP_LIMIT) {
				m_RangeMax = rngMax;
				m_RangeMin = rngMin;
				rangeSize = m_RangeMax - m_RangeMin;

				if (m_Settings.TrendRangeAuto) {
					m_TrendMax = (int)(m_RangeMax + (rangeSize * CGlobal.RANGE_MARGIN_MULTI) + 0.5);
					m_TrendMin = (int)(m_RangeMin - (rangeSize * CGlobal.RANGE_MARGIN_MULTI) - 0.5);
					m_TrendSize = m_TrendMax - m_TrendMin + 1;
				} else {
					m_TrendMax = (int)(m_Settings.TrendRangeMax);
					m_TrendMin = (int)(m_Settings.TrendRangeMin);
				}
				m_TrendSize = m_TrendMax - m_TrendMin + 1;

				double	fMinY, fMaxY;

				fMinY = m_RangeMin - m_TrendMin;
				fMaxY = m_RangeMax - m_TrendMin;

				m_MinY = m_TrendHeight - (int)((double)m_TrendHeight * fMinY / m_TrendSize + 0.5);
				m_MaxY = m_TrendHeight - (int)((double)m_TrendHeight * fMaxY / m_TrendSize + 0.5);
			}
			graph.FillRectangle(Brushes.White, 0, 0, m_TrendBmp.Width, m_TrendBmp.Height);
			graph.DrawLine(m_gridPen, 0, m_MaxY, IMG_WND_WIDTH, m_MaxY);
			graph.DrawLine(m_gridPen, 0, m_MinY, IMG_WND_WIDTH, m_MinY);
			graph.DrawRectangle(m_gridPen, 0, 0, IMG_WND_WIDTH, m_TrendHeight);

            str = m_TrendMax.ToString("F1", CultureInfo.InvariantCulture);
            size = graph.MeasureString(str, m_TrendFnt);
			graph.DrawString(str, m_TrendFnt, Brushes.Black, IMG_WND_WIDTH + 2, 0);

            str = m_TrendMin.ToString("F1", CultureInfo.InvariantCulture);
            size = graph.MeasureString(str, m_TrendFnt);
			graph.DrawString(str, m_TrendFnt, Brushes.Black, IMG_WND_WIDTH + 2, (int)(m_TrendHeight - (size.Height / 2)));

			for (i = 0; i < IMG_WND_WIDTH; i += 50) {
				x = IMG_WND_WIDTH - i;
				graph.DrawLine(m_gridPen, x, 0, x, m_TrendHeight);
				if (i > 0) {
					str = (i * -1).ToString("D", CultureInfo.InvariantCulture);
				}else {
					str = i.ToString("D", CultureInfo.InvariantCulture);
				}
				size = graph.MeasureString(str, m_TrendFnt);
				graph.DrawString(str, m_TrendFnt, Brushes.Black,
						(x - (size.Width / 2)), (int)(m_TrendHeight + 1));
			}
			//graph.DrawString(" frame", m_TrendFnt, Brushes.Black, IMG_WND_WIDTH + 2, (int)(m_TrendHeight + 3));

			graph.DrawString("  Temp", m_TrendFnt, Brushes.Black, IMG_WND_WIDTH + m_TrendValX, 0);
			graph.DrawString("( X , Y )", m_TrendFnt, Brushes.Black, IMG_WND_WIDTH + m_TrendPosX, 0);

			if (m_PntValNum < PNT_VAL_NUM) {
				idx = 0;
			} else {
				idx = m_PntValIdx;
				if (idx >= PNT_VAL_NUM) {
					idx = 0;
				}
			}
			if (showMaxVal) {
				Draw_TrendLine(graph, CGlobal.PNT_MAX_IDX, idx);
			}
			for (i = 0; i < 3; i++) {
				if (gPoint[i].X >= 0 && gPoint[i].Y >= 0) {
					Draw_TrendLine(graph, CGlobal.PNT_PNT1_IDX + i, idx);
				}
			}
			if (showMinVal) {
				Draw_TrendLine(graph, CGlobal.PNT_MIN_IDX, idx);
			}
			pictTrend.Image = m_TrendBmp;

			graph.Dispose();
		}

		private void MenuItemSettings_Click(object sender, EventArgs e) {
			FormSetting frm = new FormSetting();

			frm.m_Settings = m_Settings;
			if (frm.ShowDialog() == DialogResult.OK) {
				CGlobal.stSettings old = m_Settings;
				m_Settings = frm.m_Settings;

				if ((old.TrendRangeAuto != m_Settings.TrendRangeAuto) ||
						(old.TrendRangeMax != m_Settings.TrendRangeMax) ||
						(old.TrendRangeMin != m_Settings.TrendRangeMin)) {
					m_RangeMax = fMIN_TEMP_LIMIT;
					m_RangeMin = fMAX_TEMP_LIMIT;

					m_PntValNum = 0;
					m_PntValIdx = 0;
				}
                Invalidate();//redraw form
			}
		}

		private void BtnClearAlarm_Click(object sender, EventArgs e) {
			int	i;

			for (i = 0; i < 3; i++) {
				m_bAlarmStat[i] = false;
			}
		}

		private void Timer1_Tick(object sender, EventArgs e) {
			if (m_bAlarmBeep && m_bAlarmEnable) {
				if (m_bPlayBeep) {
					m_bPlayBeep = false;
				} else {
					Console.Beep(440, 100);
					m_bPlayBeep = true;
				}
			} else {
				m_bPlayBeep = false;
			}
		}

		private void BtnStopAlarm_Click(object sender, EventArgs e) {
			if (m_bAlarmEnable) {
				m_bAlarmEnable = false;
				BtnStopAlarm.Text = "Start Ararm";
			} else {
				m_bAlarmEnable = true;
				BtnStopAlarm.Text = "Stop Ararm";
			}
		}

		private void BtnRecord_Click(object sender, EventArgs e) {
			if (m_isRecording) {
				m_isRecording = false;
				if (m_bCsvFileOpen) {
					m_CsvFile.Close();
					m_bCsvFileOpen = false;
				}
				if (m_bAviFileOpen) {
                    m_AviFile.Dispose();
					m_bAviFileOpen = false;
				}
				BtnRecord.Text = "Start Recording";
			} else {
				DateTime	dt = DateTime.Now;
				bool		bCsvFile = false;
				bool		bAviFile = false;

				m_RecFrameCount = Decimal.ToInt32(num_RecFrameCount.Value);
				m_FrameCount = 0;
				lbl_FrameCount.Text = m_FrameCount.ToString("0");

				if (chk_CsvFile.Checked && m_Settings.TempFileName != "") {
					string	sBasename = System.IO.Path.GetFileNameWithoutExtension(m_Settings.TempFileName);
					string	sFname = m_Settings.DataSaveDir + @"\" + sBasename + 
                                            dt.ToString("_yyyyMMdd_HHmmss") + "." + CGlobal.TEMP_FILE_EXT;
					try {
						m_CsvFile = new StreamWriter(sFname, false, System.Text.Encoding.UTF8);
					} catch (Exception ex) {
						MessageBox.Show(CGlobal.TEMP_FILE_TYPE + " open error:" + ex.Message);
						return;
					}
					string	strLine;

					strLine = "Date, Time, Min, Max, Point1, Point2, Point3";
					m_CsvFile.WriteLine(strLine);

					bCsvFile = true;
				}
				if (chk_AviFile.Checked && m_Settings.VideoFileName != "") {
					string	sBasename = System.IO.Path.GetFileNameWithoutExtension(m_Settings.VideoFileName);
					string	sFname = m_Settings.DataSaveDir + @"\" + sBasename + 
                                            dt.ToString("_yyyyMMdd_HHmmss") + "." + CGlobal.VIDEO_FILE_EXT;
					try {
                        m_AviFile = new VideoWriter(sFname, FourCC.MJPG, 9.0,
                                                    new OpenCvSharp.Size(bigBitmap.Width, bigBitmap.Height));
					} catch (Exception ex) {
						MessageBox.Show(CGlobal.VIDEO_FILE_TYPE + " open error:" + ex.Message);
						if (bCsvFile) {
							m_CsvFile.Close();
							m_CsvFile.Dispose();
						}
						return;
					}
					bAviFile = true;
				}
				if (bCsvFile || bAviFile) {
					m_bCsvFileOpen = bCsvFile;
					m_bAviFileOpen = bAviFile;
					m_isRecording = true;
					BtnRecord.Text = "Stop Recording";
				} else {
					m_isRecording = false;
					BtnRecord.Text = "Start Recording";
				}
			}
		}

		private void Form1_Paint(object sender, PaintEventArgs e)
        {
            if (lastUsableFrame != null) {
                string minTemp = RawToTemp(gModeLeft);
                string maxTemp = RawToTemp(gModeRight);

                lblSliderMin.Text = minTemp;
                lblSliderMax.Text = maxTemp;
                lblMinTemp.Text = minTemp;
                lblMaxTemp.Text = maxTemp;

                if (autoRange) {	//set sliders position
                    TrackRangeMin.Value = gModeLeft;
                    TrackRangeMax.Value = gModeRight;
                }

                bitmap_data = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.WriteOnly, PixelFormat.Format24bppRgb);
                Marshal.Copy(imgBuffer, 0, bitmap_data.Scan0, imgBuffer.Length);
                bitmap.UnlockBits(bitmap_data);

                //crop image to 
                croppedBitmap = cropFilter.Apply(bitmap);

                //upscale 200 %
                bigBitmap = bilinearResize.Apply(croppedBitmap);

                //sharpen image
                if (SharpenImage) {
                    sfilter.ApplyInPlace(bigBitmap);
                }

                System.Drawing.Point[]	curPnt = new System.Drawing.Point[3];
                System.Drawing.Point	minPnt;
                System.Drawing.Point	maxPnt;
				int		i;

                switch (gTurnAngle) {
                    case 90:
                        bigBitmap.RotateFlip(RotateFlipType.Rotate90FlipNone);
                        break;
                    case 180:
                        bigBitmap.RotateFlip(RotateFlipType.Rotate180FlipNone);
                        break;
                    case 270:
                        bigBitmap.RotateFlip(RotateFlipType.Rotate270FlipNone);
                        break;
                    default:
                        break;

                }
                if (gTurnAngle > 0) {
					for (i = 0; i < 3; i++) {
						if (gPoint[i].X >= 0 && gPoint[i].Y >= 0) {
							curPnt[i] = RotatePoint(gPoint[i], croppedBitmap.Width, croppedBitmap.Height, gTurnAngle);
						} else {
							curPnt[i] = gPoint[i];
						}
					}
                    minPnt = RotatePoint(gMinPnt, croppedBitmap.Width, croppedBitmap.Height, gTurnAngle);
                    maxPnt = RotatePoint(gMaxPnt, croppedBitmap.Width, croppedBitmap.Height, gTurnAngle);
                } else {
                    curPnt = gPoint;
                    minPnt = gMinPnt;
                    maxPnt = gMaxPnt;
                }
                Graphics	graph = Graphics.FromImage(bigBitmap);
                String	str;
                int		x, y;
                int		strX, strY;
                SizeF	size;

                //指定点の温度を表示
				for (i = 0; i < 3; i++) {
					if (curPnt[i].X >= 0 && curPnt[i].Y >= 0) {
                        x = (int)(curPnt[i].X * m_fImgScaleW);
                        y = (int)(curPnt[i].Y * m_fImgScaleH);

                        graph.DrawLine(m_ShadowPen, x - 5, y + 2, x + 5, y + 2);
                        graph.DrawLine(m_ShadowPen, x + 2, y - 5, x + 2, y + 5);

                        graph.DrawLine(m_PointPen, x - 5, y, x + 5, y);
                        graph.DrawLine(m_PointPen, x, y - 5, x, y + 5);

                        str = String.Format("({0:0}){1:0.0}", i + 1, gfPntTempVal[i]);
                        size = graph.MeasureString(str, m_PntFnt);

                        if (x + size.Width + 4 > bigBitmap.Width)
                        {
                            strX = x - 4 - (int)size.Width;
                        }
                        else
                        {
                            strX = x + 4;
                        }
                        if (y + size.Height + 4 > bigBitmap.Height)
                        {
                            strY = y - 4 - (int)size.Height;
                        }
                        else
                        {
                            strY = y + 4;
                        }
                        graph.DrawString(str, m_PntFnt, Brushes.Black, strX + 2, strY + 2);
                        graph.DrawString(str, m_PntFnt, Brushes.White, strX, strY);
					}
				}
                //最低温度を表示
                if (showMinVal)
                {
                    str = gfMinTempVal.ToString("F1", CultureInfo.InvariantCulture);
                    size = graph.MeasureString(str, m_PntFnt);
                    x = minPnt.X * 2;
                    y = minPnt.Y * 2;

                    graph.DrawLine(m_ShadowPen, x - 5, y + 2, x + 5, y + 2);
                    graph.DrawLine(m_ShadowPen, x + 2, y - 5, x + 2, y + 5);

                    graph.DrawLine(m_PointPen, x - 5, y, x + 5, y);
                    graph.DrawLine(m_PointPen, x, y - 5, x, y + 5);

                    if (x + size.Width + 4 > bigBitmap.Width)
                    {
                        strX = x - 4 - (int)size.Width;
                    }
                    else
                    {
                        strX = x + 4;
                    }
                    if (y + size.Height + 4 > bigBitmap.Height)
                    {
                        strY = y - 4 - (int)size.Height;
                    }
                    else
                    {
                        strY = y + 4;
                    }
                    graph.DrawString(str, m_PntFnt, Brushes.Black, strX + 2, strY + 2);
                    graph.DrawString(str, m_PntFnt, Brushes.White, strX, strY);
                }
                //最高温度を表示
                if (showMaxVal)
                {
                    str = gfMaxTempVal.ToString("F1", CultureInfo.InvariantCulture);
                    size = graph.MeasureString(str, m_PntFnt);
                    x = maxPnt.X * 2;
                    y = maxPnt.Y * 2;

                    graph.DrawLine(m_ShadowPen, x - 5, y + 2, x + 5, y + 2);
                    graph.DrawLine(m_ShadowPen, x + 2, y - 5, x + 2, y + 5);

                    graph.DrawLine(m_PointPen, x - 5, y, x + 5, y);
                    graph.DrawLine(m_PointPen, x, y - 5, x, y + 5);

                    if (x + size.Width + 4 > bigBitmap.Width)
                    {
                        strX = x - 4 - (int)size.Width;
                    }
                    else
                    {
                        strX = x + 4;
                    }
                    if (y + size.Height + 4 > bigBitmap.Height)
                    {
                        strY = y - 4 - (int)size.Height;
                    }
                    else
                    {
                        strY = y + 4;
                    }
                    graph.DrawString(str, m_PntFnt, Brushes.Black, strX + 2, strY + 2);
                    graph.DrawString(str, m_PntFnt, Brushes.White, strX, strY);
                }
                graph.Dispose();

			    //トレンドグラフ作成用データを保存
			    m_PntValArr[CGlobal.PNT_MAX_IDX, m_PntValIdx] = gfMaxTempVal;
			    m_PntValArr[CGlobal.PNT_PNT1_IDX, m_PntValIdx] = gfPntTempVal[0];
			    m_PntValArr[CGlobal.PNT_PNT2_IDX, m_PntValIdx] = gfPntTempVal[1];
			    m_PntValArr[CGlobal.PNT_PNT3_IDX, m_PntValIdx] = gfPntTempVal[2];
			    m_PntValArr[CGlobal.PNT_MIN_IDX, m_PntValIdx] = gfMinTempVal;

			    m_bAlarmBeep = false;
                if (m_bAlarmEnable) {
			        for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
				        if (m_Settings.AlarmTemp[i].AlarmEnable) {
					        if (m_Settings.AlarmTemp[i].AlarmUpLow == CGlobal.ALARM_TYPE_OVER) {
						        if (m_PntValArr[i, m_PntValIdx] > m_Settings.AlarmTemp[i].AlarmTemp) {
							        m_bAlarmStat[i] = true;
						        } else {
							        if (m_Settings.AlarmTemp[i].AlarmHold == CGlobal.ALARM_STATUS_CLEAR) {
								        m_bAlarmStat[i] = false;
							        }
						        }
					        } else if (m_Settings.AlarmTemp[i].AlarmUpLow == CGlobal.ALARM_TYPE_UNDER) {
						        if (m_PntValArr[i, m_PntValIdx] < m_Settings.AlarmTemp[i].AlarmTemp) {
							        m_bAlarmStat[i] = true;
						        } else {
							        if (m_Settings.AlarmTemp[i].AlarmHold == CGlobal.ALARM_STATUS_CLEAR) {
								        m_bAlarmStat[i] = false;
							        }
						        }
					        }
				        }
				        if (m_bAlarmStat[i] && m_Settings.AlarmTemp[i].AlarmBeep) {
					        m_bAlarmBeep = true;
				        }
			        }
				} else {
                    for (i = 0; i < CGlobal.PNT_IDX_NUM; i++) {
                        m_bAlarmStat[i] = false;
                    }
				}
                m_PntValIdx++;
			    if (m_PntValIdx >= PNT_VAL_NUM) {
				    m_PntValIdx = 0;
			    }
			    if (m_PntValNum < PNT_VAL_NUM) {
				    m_PntValNum++;
			    }
				if (m_isRecording) {
					if (m_bCsvFileOpen) {
						string		strLine;
						DateTime	dt = DateTime.Now;

						strLine = dt.ToString("yyyy/MM/dd,HH:mm:ss.fff,");
						strLine += String.Format("{0:0.0},{1:0.0}", gfMinTempVal, gfMaxTempVal);
						for (i = 0; i < 3; i++) {
							if (gPoint[i].X >= 0 && gPoint[i].Y >= 0) {
								strLine += String.Format(",{0:0.0}", gfPntTempVal[i]);
							}
						}
						m_CsvFile.WriteLine(strLine);
					}
                    if (m_bAviFileOpen) {
                        try {
                            OpenCvSharp.Mat img = (OpenCvSharp.Mat)BitmapConverter.ToMat(bigBitmap);
                            m_AviFile.Write(img);
                        } catch (Exception) {
							m_bAviFileOpen = false;
							m_AviFile.Dispose();
						}
					}
					m_FrameCount++;
					lbl_FrameCount.Text = m_FrameCount.ToString("0");
					if (m_RecFrameCount > 0 && m_FrameCount >= m_RecFrameCount) {
    					if (m_bAviFileOpen) {
							m_bAviFileOpen = false;
							m_AviFile.Dispose();
						}
						if (m_bCsvFileOpen) {
							m_bCsvFileOpen = false;
							m_CsvFile.Close();
							m_CsvFile.Dispose();
						}
						BtnRecord.Text = "Start Recording";
						m_isRecording = false;
					}
				}
                PictImage.Image = bigBitmap;

				Draw_Trend();

                if (saveImage)
                {
					string	fname;
					string	path;

                    saveImage = false;

					fname = "seek_" + DateTime.Now.ToString("yyyyMMdd_HHmmss_fff") + ".png";
					if (!Directory.Exists(m_Settings.DataSaveDir)) {
						try {
							Directory.CreateDirectory(m_Settings.DataSaveDir);
							path = m_Settings.DataSaveDir + @"\" + fname;
						} catch (IOException) {
							path = m_sModuleDir + @"\" + fname;
						}
					} else {
						path = m_Settings.DataSaveDir + @"\" + fname;
					}
                    bigBitmap.Save(path);
                }
                //DrawHistogram();
            }
        }
    }
}
